// edit.js

import { grveSelectOptions, grveGetColorArray } from '../utilities';
import get from 'lodash/get';
import apiFetch from '@wordpress/api-fetch';
import ServerSideRender from '@wordpress/server-side-render';

import { addQueryArgs } from '@wordpress/url';
import { __ } from '@wordpress/i18n';
import { InspectorControls } from '@wordpress/block-editor';
import { QueryControls, TextControl, SelectControl, ToggleControl, PanelBody } from '@wordpress/components';
import { Component, Fragment } from '@wordpress/element';

const CATEGORIES_LIST_QUERY = {
	per_page: -1,
	hide_empty: true,
	_fields: 'id, name ,slug',
};

export default class ImpekaBlogLeaderEdit extends Component {
	constructor() {
		super( ...arguments );
		this.state = {
			categoriesList: [],

		};
	}
	componentDidMount() {
		this.isStillMounted = true;
		this.fetchRequest = apiFetch( {
			path: addQueryArgs( '/wp/v2/categories', CATEGORIES_LIST_QUERY ),

		} ).then(
			( categoriesList ) => {
				if ( this.isStillMounted ) {
					this.setState( { categoriesList } );
				}
			}
		).catch(
			() => {
				if ( this.isStillMounted ) {
					this.setState( { categoriesList: [] } );
				}
			}
		);
	}
	componentWillUnmount() {
		this.isStillMounted = false;
	}
    render() {

		const { categoriesList } = this.state;
		const { attributes, setAttributes, className } = this.props;
		const {
			posts_per_page, image_load_mode, image_mode,
			excerpt_length, excerpt_more, excerpt_more_text,
			animation, animation_delay, animation_duration,
			order_by, order, margin_bottom,
			leader_style, overlay_color, overlay_opacity,
			heading_tag, heading, small_post_heading,
			show_author, show_date, show_comments, show_like,
			exclude_posts, categories, include_posts,
		} = attributes;

		const categoryQueryControls = () => {
			const categorySuggestions = categoriesList.reduce(
				( accumulator, category ) => ( {
					...accumulator,
					[ category.name ]: category,
				} ),
				{}
			);
			const suggestions = categoriesList.reduce(
				( accumulator, category ) => ( {
					...accumulator,
					[ category.name ]: category,
				} ),
				{}
			);
			return ( <QueryControls
				categorySuggestions={ categorySuggestions }
				selectedCategories={ attributes.categories }
				onCategoryChange={ ( tokens ) => {
					const allCategories = tokens.map( ( token ) =>
						typeof token === 'string' ? suggestions[ token ] : token
					);
					setAttributes( { categories: allCategories } );
				} }
			/> );
		};

		return (
			<Fragment>
				<InspectorControls key="inspector">
					<PanelBody title={ __( 'General', 'impeka-gutenberg-extension' ) } initialOpen={ true } >
						<SelectControl
							label={ __( 'Number of Posts', 'impeka-gutenberg-extension' ) }
							help={ __( 'Enter how many posts you want to display.', 'impeka-gutenberg-extension' ) }
							value={ posts_per_page }
							options={
								[
									{ value: '1', label: __( 'Leader Only', 'impeka-gutenberg-extension' ) },
									{ value: '3', label: __( 'Leader + 2 columns', 'impeka-gutenberg-extension' ) },
									{ value: '4', label: __( 'Leader + 3 columns', 'impeka-gutenberg-extension' ) },
									{ value: '5', label: __( 'Leader + 4 columns', 'impeka-gutenberg-extension' ) },
								]
							}
							onChange={ val => { setAttributes( { posts_per_page: val } ); } }
						/>
						<SelectControl
							label={ __( 'Image Load Mode', 'impeka-gutenberg-extension' ) }
							help={ __( 'Enable Lazy load.', 'impeka-gutenberg-extension' ) }
							value={ image_load_mode }
							options={
								grveSelectOptions[ "image_load_mode" ]
							}
							onChange={ val => { setAttributes( { image_load_mode: val } ); } }
						/>
						<SelectControl
							label={ __( 'Image Mode', 'impeka-gutenberg-extension' ) }
							help={ __( 'Select your Image Mode.', 'impeka-gutenberg-extension' ) }
							value={ image_mode }
							options={
								[
									{ value: 'landscape', label: __( 'Landscape Small Crop', 'impeka-gutenberg-extension' ) },
									{ value: 'large', label: __( 'Resize ( Large )', 'impeka-gutenberg-extension' ) },
									{ value: 'medium_large', label: __( 'Resize ( Medium Large )', 'impeka-gutenberg-extension' ) },
									{ value: 'medium', label: __( 'Resize ( Medium )', 'impeka-gutenberg-extension' ) },
								]
							}
							onChange={ val => { setAttributes( { image_mode: val } ); } }
						/>
						<TextControl
							label={ __( 'Excerpt length', 'impeka-gutenberg-extension' ) }
							help={ __( 'Type how many words you want to display in your post excerpts.', 'impeka-gutenberg-extension' ) }
							value={ excerpt_length }
							onChange={ val => { setAttributes( { excerpt_length: val } ); } }
						/>
						<ToggleControl
							label={ __( 'Read More', 'impeka-gutenberg-extension' ) }
							help={ __( 'Adds a read more button after the excerpt.', 'impeka-gutenberg-extension' ) }
							checked={ excerpt_more }
							onChange={ val => { setAttributes( { excerpt_more: ! excerpt_more } ); } }
						/>
						{ excerpt_more && (
						<TextControl
							label={ __( 'Read more text', 'impeka-gutenberg-extension' ) }
							help={ __( 'Type your read more text.', 'impeka-gutenberg-extension' ) }
							value={ excerpt_more_text }
							onChange={ val => { setAttributes( { excerpt_more_text: val } ); } }
						/>
						)}
						<SelectControl
							label={ __( 'Animation', 'impeka-gutenberg-extension' ) }
							help={ __( 'Select type of animation if you want this element to be animated when it enters into the browsers viewport. Note: Works only in modern browsers.', 'impeka-gutenberg-extension' ) }
							value={ animation }
							options={
								grveSelectOptions[ "animation" ]
							}
							onChange={ val => { setAttributes( { animation: val } ); } }
						/>
						{ animation && (
						<TextControl
							label={ __( 'Animation Delay', 'impeka-gutenberg-extension' ) }
							help={ __( 'Add delay in milliseconds.', 'impeka-gutenberg-extension' ) }
							value={ animation_delay }
							onChange={ val => { setAttributes( { animation_delay: val } ); } }
						/>
						)}
						{ animation && (
						<SelectControl
							label={ __( 'Animation Duration', 'impeka-gutenberg-extension' ) }
							help={ __( 'Select the duration for your animated element.', 'impeka-gutenberg-extension' ) }
							value={ animation_duration }
							options={
								grveSelectOptions[ "animation_duration" ]
							}
							onChange={ val => { setAttributes( { animation_duration: val } ); } }
						/>
						)}
						<SelectControl
							label={ __( 'Order by', 'impeka-gutenberg-extension' ) }
							value={ order_by }
							options={
								grveSelectOptions[ "order_by" ]
							}
							onChange={ val => { setAttributes( { order_by: val } ); } }
						/>
						{ ( 'rand' != order_by && 'post__in' != order_by ) && (
						<SelectControl
							label={ __( 'Order', 'impeka-gutenberg-extension' ) }
							value={ order }
							options={
								grveSelectOptions[ "order" ]
							}
							onChange={ val => { setAttributes( { order: val } ); } }
						/>
						)}
						<TextControl
							label={ __( 'Bottom margin', 'impeka-gutenberg-extension' ) }
							help={ __( 'You can use px, em, %, etc. or enter just number and it will use pixels.', 'impeka-gutenberg-extension' ) }
							value={ margin_bottom }
							onChange={ val => { setAttributes( { margin_bottom: val } ); } }
						/>
					</PanelBody>
					<PanelBody title={ __( 'Titles & Styles', 'impeka-gutenberg-extension' ) } initialOpen={ false } >
						<SelectControl
							label={ __( 'Leader Style', 'impeka-gutenberg-extension' ) }
							help={ __( 'Select your Post Leader Style.', 'impeka-gutenberg-extension' ) }
							value={ leader_style }
							options={
								[
									{ value: '', label: __( 'Horizontal', 'impeka-gutenberg-extension' ) },
									{ value: 'vertical', label: __( 'Vertical', 'impeka-gutenberg-extension' ) },
								]
							}
							onChange={ val => { setAttributes( { leader_style: val } ); } }
						/>
						<SelectControl
							label={ __( 'Leader Background Color', 'impeka-gutenberg-extension' ) }
							help={ __( 'This affects the Background of the item.', 'impeka-gutenberg-extension' ) }
							value={ overlay_color }
							options={ grveGetColorArray( ['default', 'primary', 'gradient', 'standard'] ) }
							onChange={ val => { setAttributes( { overlay_color: val } ); } }
						/>
						<SelectControl
							label={ __( 'Leader Background Opacity', 'impeka-gutenberg-extension' ) }
							help={ __( 'Choose the opacity for the overlay.', 'impeka-gutenberg-extension' ) }
							value={ overlay_opacity }
							options={ grveSelectOptions[ "overlay_opacity" ] }
							onChange={ val => { setAttributes( { overlay_opacity: val } ); } }
						/>
						<SelectControl
							label={ __( 'Post Title Tag', 'impeka-gutenberg-extension' ) }
							help={ __( 'Post Title Tag for SEO.', 'impeka-gutenberg-extension' ) }
							value={ heading_tag }
							options={ grveSelectOptions[ "heading_tag" ] }
							onChange={ val => { setAttributes( { heading_tag: val } ); } }
						/>
						<SelectControl
							label={ __( 'Leader Post Title Size/Typography', 'impeka-gutenberg-extension' ) }
							help={ __( 'Leader Post Title size and typography, defined in Theme Options - Typography Options', 'impeka-gutenberg-extension' ) }
							value={ heading }
							options={ grveSelectOptions[ "heading" ] }
							onChange={ val => { setAttributes( { heading: val } ); } }
						/>
						<SelectControl
							label={ __( 'Small Post Titles Size/Typography', 'impeka-gutenberg-extension' ) }
							help={ __( 'Small Post Titles size and typography, defined in Theme Options - Typography Options', 'impeka-gutenberg-extension' ) }
							value={ small_post_heading }
							options={ grveSelectOptions[ "heading" ] }
							onChange={ val => { setAttributes( { small_post_heading: val } ); } }
						/>
					</PanelBody>
					<PanelBody title={ __( 'Extras', 'impeka-gutenberg-extension' ) } initialOpen={ false } >
						<ToggleControl
							label={ __( 'Author Field Visibility', 'impeka-gutenberg-extension' ) }
							help={ __( 'Enable or Disable the author field of your blog overview.', 'impeka-gutenberg-extension' ) }
							checked={ show_author }
							onChange={ val => { setAttributes( { show_author: ! show_author } ); } }
						/>
						<ToggleControl
							label={ __( 'Date Field Visibility', 'impeka-gutenberg-extension' ) }
							help={ __( 'Enable or Disable the date field of your blog overview.', 'impeka-gutenberg-extension' ) }
							checked={ show_date }
							onChange={ val => { setAttributes( { show_date: ! show_date } ); } }
						/>
						<ToggleControl
							label={ __( 'Comments Field Visibility', 'impeka-gutenberg-extension' ) }
							help={ __( 'Enable or Disable the comments field of your blog overview.', 'impeka-gutenberg-extension' ) }
							checked={ show_comments }
							onChange={ val => { setAttributes( { show_comments: ! show_comments } ); } }
						/>
						<ToggleControl
							label={ __( 'Like Field Visibility', 'impeka-gutenberg-extension' ) }
							help={ __( 'Enable or Disable the like field of your blog overview.', 'impeka-gutenberg-extension' ) }
							checked={ show_like }
							onChange={ val => { setAttributes( { show_like: ! show_like } ); } }
						/>
					</PanelBody>
					<PanelBody title={ __( 'Categories', 'impeka-gutenberg-extension' ) } initialOpen={ false } >
						<TextControl
							label={ __( 'Exclude Posts', 'impeka-gutenberg-extension' ) }
							help={ __( 'Type the post ids you want to exclude separated by comma ( , ).', 'impeka-gutenberg-extension' ) }
							value={ exclude_posts }
							onChange={ val => { setAttributes( { exclude_posts: val } ); } }
						/>
						{ categoryQueryControls() }
						<TextControl
							label={ __( 'Include Posts', 'impeka-gutenberg-extension' ) }
							help={ __( 'Type the specific post ids you want to include separated by comma ( , ). Note: If you define specific post ids, Exclude Posts and Categories will have no effect.', 'impeka-gutenberg-extension' ) }
							value={ include_posts }
							onChange={ val => { setAttributes( { include_posts: val } ); } }
						/>
					</PanelBody>
				</InspectorControls>
				<ServerSideRender block="impeka/blog-leader" attributes={ attributes } />
			</Fragment>
		);
    }
}