// edit.js

import { grveSelectOptions, grveGetColorArray } from '../utilities';
import get from 'lodash/get';
import apiFetch from '@wordpress/api-fetch';
import ServerSideRender from '@wordpress/server-side-render';

import { addQueryArgs } from '@wordpress/url';
import { __ } from '@wordpress/i18n';
import { InspectorControls } from '@wordpress/block-editor';
import { QueryControls, TextControl, SelectControl, ToggleControl, PanelBody } from '@wordpress/components';
import { Component, Fragment } from '@wordpress/element';

const CATEGORIES_LIST_QUERY = {
	per_page: -1,
	hide_empty: true,
	_fields: 'id, name ,slug',
};

export default class ImpekaListTitleEdit extends Component {
	constructor() {
		super( ...arguments );
		this.state = {
			categoriesList: [],
			portfolioCategoriesList: [],
			productCategoriesList: [],
		};
	}
	componentDidMount() {
		this.isStillMounted = true;
		//Post
		this.fetchRequest = apiFetch( {
			path: addQueryArgs( '/wp/v2/categories', CATEGORIES_LIST_QUERY ),

		} ).then(
			( categoriesList ) => {
				if ( this.isStillMounted ) {
					this.setState( { categoriesList } );
				}
			}
		).catch(
			() => {
				if ( this.isStillMounted ) {
					this.setState( { categoriesList: [] } );
				}
			}
		);
		//Portfolio
		this.fetchRequest = apiFetch( {
			path: addQueryArgs( '/wp/v2/portfolio_category', CATEGORIES_LIST_QUERY ),

		} ).then(
			( portfolioCategoriesList ) => {
				if ( this.isStillMounted ) {
					this.setState( { portfolioCategoriesList } );
				}
			}
		).catch(
			() => {
				if ( this.isStillMounted ) {
					this.setState( { portfolioCategoriesList: [] } );
				}
			}
		);
		//Product
		this.fetchRequest = apiFetch( {
			path: addQueryArgs( '/wc/v2/products/categories', CATEGORIES_LIST_QUERY ),

		} ).then(
			( productCategoriesList ) => {
				if ( this.isStillMounted ) {
					this.setState( { productCategoriesList } );
				}
			}
		).catch(
			() => {
				if ( this.isStillMounted ) {
					this.setState( { productCategoriesList: [] } );
				}
			}
		);
	}
	componentWillUnmount() {
		this.isStillMounted = false;
	}
    render() {

		const { categoriesList, portfolioCategoriesList, productCategoriesList } = this.state;
		const { attributes, setAttributes, className } = this.props;
		const {
			post_type, order_by, order, margin_bottom,
			overlay_color, overlay_opacity,
			heading_tag, heading, title_color, text_color, line_color,
			posts_per_page, display_style,
			filter, filter_all_text, filter_order_by, filter_order, filter_style, filter_shape, filter_color, filter_align, filter_gototop,
			exclude_posts, categories, portfolio_categories, products_categories, include_posts,
		} = attributes;

		const categoryQueryControls = () => {
			const categorySuggestions = categoriesList.reduce(
				( accumulator, category ) => ( {
					...accumulator,
					[ category.name ]: category,
				} ),
				{}
			);
			const suggestions = categoriesList.reduce(
				( accumulator, category ) => ( {
					...accumulator,
					[ category.name ]: category,
				} ),
				{}
			);
			return ( <QueryControls
				categorySuggestions={ categorySuggestions }
				selectedCategories={ attributes.categories }
				onCategoryChange={ ( tokens ) => {
					const allCategories = tokens.map( ( token ) =>
						typeof token === 'string' ? suggestions[ token ] : token
					);
					setAttributes( { categories: allCategories } );
				} }
			/> );
		};

		const portfolioCategoryQueryControls = () => {
			const categorySuggestions = portfolioCategoriesList.reduce(
				( accumulator, category ) => ( {
					...accumulator,
					[ category.name ]: category,
				} ),
				{}
			);
			const suggestions = portfolioCategoriesList.reduce(
				( accumulator, category ) => ( {
					...accumulator,
					[ category.name ]: category,
				} ),
				{}
			);
			return ( <QueryControls
				categorySuggestions={ categorySuggestions }
				selectedCategories={ attributes.portfolio_categories }
				onCategoryChange={ ( tokens ) => {
					const allCategories = tokens.map( ( token ) =>
						typeof token === 'string' ? suggestions[ token ] : token
					);
					setAttributes( { portfolio_categories: allCategories } );
				} }
			/> );
		};

		const productCategoryQueryControls = () => {
			const categorySuggestions = productCategoriesList.reduce(
				( accumulator, category ) => ( {
					...accumulator,
					[ category.name ]: category,
				} ),
				{}
			);
			const suggestions = productCategoriesList.reduce(
				( accumulator, category ) => ( {
					...accumulator,
					[ category.name ]: category,
				} ),
				{}
			);
			return ( <QueryControls
				categorySuggestions={ categorySuggestions }
				selectedCategories={ attributes.product_categories }
				onCategoryChange={ ( tokens ) => {
					const allCategories = tokens.map( ( token ) =>
						typeof token === 'string' ? suggestions[ token ] : token
					);
					setAttributes( { product_categories: allCategories } );
				} }
			/> );
		};

		return (
			<Fragment>
				<InspectorControls key="inspector">
					<PanelBody title={ __( 'General', 'impeka-gutenberg-extension' ) } initialOpen={ true } >
						<SelectControl
							label={ __( 'Post Type', 'impeka-gutenberg-extension' ) }
							help={ __( 'Select your Post Type.', 'impeka-gutenberg-extension' ) }
							value={ post_type }
							options={
								[
									{ value: 'post', label: __( 'Post', 'impeka-gutenberg-extension' ) },
									{ value: 'portfolio', label: __( 'Portfolio', 'impeka-gutenberg-extension' ) },
									{ value: 'product', label: __( 'Product', 'impeka-gutenberg-extension' ) },
								]
							}
							onChange={ val => { setAttributes( { post_type: val } ); } }
						/>
						<SelectControl
							label={ __( 'Order by', 'impeka-gutenberg-extension' ) }
							value={ order_by }
							options={
								grveSelectOptions[ "order_by" ]
							}
							onChange={ val => { setAttributes( { order_by: val } ); } }
						/>
						{ ( 'rand' != order_by && 'post__in' != order_by ) && (
						<SelectControl
							label={ __( 'Order', 'impeka-gutenberg-extension' ) }
							value={ order }
							options={
								grveSelectOptions[ "order" ]
							}
							onChange={ val => { setAttributes( { order: val } ); } }
						/>
						)}
						<TextControl
							label={ __( 'Bottom margin', 'impeka-gutenberg-extension' ) }
							help={ __( 'You can use px, em, %, etc. or enter just number and it will use pixels.', 'impeka-gutenberg-extension' ) }
							value={ margin_bottom }
							onChange={ val => { setAttributes( { margin_bottom: val } ); } }
						/>
					</PanelBody>
					<PanelBody title={ __( 'Styles', 'impeka-gutenberg-extension' ) } initialOpen={ false } >
						<SelectControl
							label={ __( 'Overlay Color', 'impeka-gutenberg-extension' ) }
							help={ __( 'Choose the overlay color.', 'impeka-gutenberg-extension' ) }
							value={ overlay_color }
							options={ grveGetColorArray( ['default', 'primary', 'gradient', 'standard'] ) }
							onChange={ val => { setAttributes( { overlay_color: val } ); } }
						/>
						<SelectControl
							label={ __( 'Overlay Opacity', 'impeka-gutenberg-extension' ) }
							help={ __( 'Choose the opacity for the overlay.', 'impeka-gutenberg-extension' ) }
							value={ overlay_opacity }
							options={ grveSelectOptions[ "overlay_opacity" ] }
							onChange={ val => { setAttributes( { overlay_opacity: val } ); } }
						/>
						<SelectControl
							label={ __( 'Title Color', 'impeka-gutenberg-extension' ) }
							help={ __( 'Choose the title color.', 'impeka-gutenberg-extension' ) }
							value={ title_color }
							options={ grveGetColorArray( ['default', 'primary', 'standard'] ) }
							onChange={ val => { setAttributes( { title_color: val } ); } }
						/>
						<SelectControl
							label={ __( 'Text Color', 'impeka-gutenberg-extension' ) }
							help={ __( 'Choose the text color.', 'impeka-gutenberg-extension' ) }
							value={ text_color }
							options={ grveGetColorArray( ['default', 'primary', 'standard'] ) }
							onChange={ val => { setAttributes( { text_color: val } ); } }
						/>
						<SelectControl
							label={ __( 'Line Color', 'impeka-gutenberg-extension' ) }
							help={ __( 'Choose the line color', 'impeka-gutenberg-extension' ) }
							value={ line_color }
							options={ grveGetColorArray( ['primary', 'standard'] ) }
							onChange={ val => { setAttributes( { line_color: val } ); } }
						/>
					</PanelBody>
					<PanelBody title={ __( 'Titles', 'impeka-gutenberg-extension' ) } initialOpen={ false } >
						<SelectControl
							label={ __( 'Title Tag', 'impeka-gutenberg-extension' ) }
							help={ __( 'Title Tag for SEO.', 'impeka-gutenberg-extension' ) }
							value={ heading_tag }
							options={ grveSelectOptions[ "heading_tag" ] }
							onChange={ val => { setAttributes( { heading_tag: val } ); } }
						/>
						<SelectControl
							label={ __( 'Title Size/Typography', 'impeka-gutenberg-extension' ) }
							help={ __( 'Title size and typography, defined in Theme Options - Typography Options', 'impeka-gutenberg-extension' ) }
							value={ heading }
							options={ grveSelectOptions[ "heading" ] }
							onChange={ val => { setAttributes( { heading: val } ); } }
						/>
					</PanelBody>
					<PanelBody title={ __( 'Pagination', 'impeka-gutenberg-extension' ) } initialOpen={ false } >
						<TextControl
							label={ __( 'Items per page', 'impeka-gutenberg-extension' ) }
							help={ __( 'Enter how many items per page you want to display.', 'impeka-gutenberg-extension' ) }
							value={ posts_per_page }
							onChange={ val => { setAttributes( { posts_per_page: val } ); } }
						/>
						<SelectControl
							label={ __( 'Display Style', 'impeka-gutenberg-extension' ) }
							help={ __( 'Select Show All or define your pagination style.', 'impeka-gutenberg-extension' ) }
							value={ display_style }
							options={ grveSelectOptions[ "display_simple_style" ] }
							onChange={ val => { setAttributes( { display_style: val } ); } }
						/>
					</PanelBody>
					<PanelBody title={ __( 'Filters', 'impeka-gutenberg-extension' ) } initialOpen={ false } >
						<ToggleControl
							label={ __( 'Filter', 'impeka-gutenberg-extension' ) }
							help={ __( 'If selected, an isotope filter will be displayed.', 'impeka-gutenberg-extension' ) }
							checked={ filter }
							onChange={ val => { setAttributes( { filter: ! filter } ); } }
						/>
						{ filter && (
						<Fragment>
						<TextControl
							label={ __( 'Filter All Text', 'impeka-gutenberg-extension' ) }
							help={ __( 'Type the text to show all filters ( Default is: All ).', 'impeka-gutenberg-extension' ) }
							value={ filter_all_text }
							onChange={ val => { setAttributes( { filter_all_text: val } ); } }
						/>
						<SelectControl
							label={ __( 'Filter Order By', 'impeka-gutenberg-extension' ) }
							value={ filter_order_by }
							options={ grveSelectOptions[ "filter_order_by" ] }
							onChange={ val => { setAttributes( { filter_order_by: val } ); } }
						/>
						<SelectControl
							label={ __( 'Filter Order', 'impeka-gutenberg-extension' ) }
							value={ filter_order }
							options={ grveSelectOptions[ "filter_order" ] }
							onChange={ val => { setAttributes( { filter_order: val } ); } }
						/>
						<SelectControl
							label={ __( 'Filter Style', 'impeka-gutenberg-extension' ) }
							value={ filter_style }
							options={ grveSelectOptions[ "filter_style" ] }
							onChange={ val => { setAttributes( { filter_style: val } ); } }
						/>
						<SelectControl
							label={ __( 'Filter Shape', 'impeka-gutenberg-extension' ) }
							value={ filter_shape }
							options={ grveSelectOptions[ "filter_shape" ] }
							onChange={ val => { setAttributes( { filter_shape: val } ); } }
						/>
						<SelectControl
							label={ __( 'Filter Color', 'impeka-gutenberg-extension' ) }
							value={ filter_color }
							options={ grveGetColorArray() }
							onChange={ val => { setAttributes( { filter_color: val } ); } }
						/>
						<SelectControl
							label={ __( 'Filter Align', 'impeka-gutenberg-extension' ) }
							value={ filter_align }
							options={ grveSelectOptions[ "filter_align" ] }
							onChange={ val => { setAttributes( { filter_align: val } ); } }
						/>
						<SelectControl
							label={ __( 'Filter Go To Top', 'impeka-gutenberg-extension' ) }
							value={ filter_gototop }
							options={ grveSelectOptions[ "filter_gototop" ] }
							onChange={ val => { setAttributes( { filter_gototop: val } ); } }
						/>
						</Fragment>
						)}
					</PanelBody>
					<PanelBody title={ __( 'Categories', 'impeka-gutenberg-extension' ) } initialOpen={ false } >
						<TextControl
							label={ __( 'Exclude Posts', 'impeka-gutenberg-extension' ) }
							help={ __( 'Type the post ids you want to exclude separated by comma ( , ).', 'impeka-gutenberg-extension' ) }
							value={ exclude_posts }
							onChange={ val => { setAttributes( { exclude_posts: val } ); } }
						/>
						{ 'post' == post_type && ( categoryQueryControls() )}
						{ 'portfolio' == post_type && ( portfolioCategoryQueryControls() )}
						{ 'product' == post_type && ( productCategoryQueryControls() )}
						<TextControl
							label={ __( 'Include Posts', 'impeka-gutenberg-extension' ) }
							help={ __( 'Type the specific post ids you want to include separated by comma ( , ). Note: If you define specific post ids, Exclude Posts and Categories will have no effect.', 'impeka-gutenberg-extension' ) }
							value={ include_posts }
							onChange={ val => { setAttributes( { include_posts: val } ); } }
						/>
					</PanelBody>
				</InspectorControls>
				<ServerSideRender block="impeka/list-title" attributes={ attributes } />
			</Fragment>
		);
    }
}