<?php
/**
 * Impeka Importer class for managing the import process of a WXR file
 *
 */

/**
 * Impeka importer class.
 */
class Impeka_Importer extends WP_Importer {
	var $max_wxr_version = 1.2; // max. supported WXR version

	var $id; // WXR attachment ID

	// information to import from WXR file
	var $version;
	var $authors = array();
	var $posts = array();
	var $terms = array();
	var $categories = array();
	var $tags = array();
	var $base_url = '';

	// mappings from old information to new
	var $processed_authors = array();
	var $author_mapping = array();
	var $processed_terms = array();
	var $processed_posts = array();
	var $post_orphans = array();
	var $processed_menu_items = array();
	var $menu_item_orphans = array();
	var $missing_menu_items = array();

	var $fetch_attachments = false;
	var $fetch_live_images = false;
	var $url_remap = array();
	var $featured_images = array();

	function __construct() {
		add_action( 'wp_ajax_impeka_import_attachments', array( $this, 'impeka_import_attachments' ) );
		add_action( 'wp_ajax_impeka_import_demo_data', array( $this, 'impeka_import_demo_data' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'impeka_demo_importer_scripts' ) );
		add_action( 'impeka_grve_demo_selection', array( $this, 'demo_theme_selection' ) );
	}

	function Impeka_Importer() {
		$this->__construct();
	}

	/**
	 * Registered callback function for the WordPress Importer
	 *
	 * Manages the three separate stages of the WXR import process
	 */
	function dispatch() {

		$step = empty( $_GET['step'] ) ? 1 : (int) $_GET['step'];
		switch ( $step ) {
			case 2:
				$this->contentSelection();
			break;
			case 3:
				$this->finish();
			break;
			case 1:
			default:
				$this->greet();
			break;
		}
	}

	/**
	 * The main controller for the actual import stage.
	 *
	 * @param string $file Path to the WXR file for importing
	 */
	function import( $file, $ids = array() ) {
		add_filter( 'import_post_meta_key', array( $this, 'is_valid_meta_key' ) );
		add_filter( 'http_request_timeout', array( &$this, 'bump_request_timeout' ) );

		if ( !empty( $ids ) ) {
			$ids_array = array();
			$ids_string = implode ( ',' , $ids  );
			$ids_array = explode( ',' , $ids_string);
			$ids_array = array_unique( $ids_array );
		} else {
			$ids_array = array();
		}

		$this->import_start( $file );

		$this->get_author_mapping();

		wp_suspend_cache_invalidation( true );
		if ( empty( $ids ) ) {
			$this->process_categories();
			$this->process_tags();
			$this->process_terms();
		}
		$this->process_posts( $ids_array );
		wp_suspend_cache_invalidation( false );

		// update incorrect/missing information in the DB
		$this->backfill_parents();
		$this->backfill_attachment_urls();
		$this->remap_featured_images();

		$this->import_end();
	}

	/**
	 * Parses the WXR file and prepares us for the task of processing parsed data
	 *
	 * @param string $file Path to the WXR file for importing
	 */
	function import_start( $file ) {
		if ( ! is_file($file) ) {
			echo '<p><strong>' . esc_html__( 'Sorry, there has been an error.', 'impeka-importer' ) . '</strong><br />';
			echo esc_html__( 'The file/demo does not exist, please try again.', 'impeka-importer' ) . '</p>';
			$this->footer();
			die();
		}

		$import_data = $this->parse( $file );

		if ( is_wp_error( $import_data ) ) {
			echo '<p><strong>' . esc_html__( 'Sorry, there has been an error.', 'impeka-importer' ) . '</strong><br />';
			echo esc_html( $import_data->get_error_message() ) . '</p>';
			$this->footer();
			die();
		}

		$this->version = $import_data['version'];
		$this->get_authors_from_import( $import_data );
		$this->posts = $import_data['posts'];
		$this->terms = $import_data['terms'];
		$this->categories = $import_data['categories'];
		$this->tags = $import_data['tags'];
		$this->base_url = esc_url( $import_data['base_url'] );

		wp_defer_term_counting( true );
		wp_defer_comment_counting( true );

		do_action( 'import_start' );
	}

	/**
	 * Performs post-import cleanup of files and the cache
	 */
	function import_end() {
		wp_import_cleanup( $this->id );

		wp_cache_flush();
		foreach ( get_taxonomies() as $tax ) {
			delete_option( "{$tax}_children" );
			_get_term_hierarchy( $tax );
		}

		wp_defer_term_counting( false );
		wp_defer_comment_counting( false );

		do_action( 'import_end' );
	}

	/**
	 * Handles the WXR upload and initial parsing of the file to prepare for
	 * displaying author import options
	 *
	 * @return bool False if error uploading or invalid file, true otherwise
	 */
	function handle_upload() {
		$file = wp_import_handle_upload();

		if ( isset( $file['error'] ) ) {
			echo '<p><strong>' . esc_html__( 'Sorry, there has been an error.', 'impeka-importer' ) . '</strong><br />';
			echo esc_html( $file['error'] ) . '</p>';
			return false;
		} else if ( ! file_exists( $file['file'] ) ) {
			echo '<p><strong>' . esc_html__( 'Sorry, there has been an error.', 'impeka-importer' ) . '</strong><br />';
			printf( esc_html__( 'The export file could not be found at %s. It is likely that this was caused by a permissions problem.', 'impeka-importer' ), esc_html( $file['file'] ) );
			echo '</p>';
			return false;
		}

		$this->id = (int) $file['id'];
		$import_data = $this->parse( $file['file'] );
		if ( is_wp_error( $import_data ) ) {
			echo '<p><strong>' . esc_html__( 'Sorry, there has been an error.', 'impeka-importer' ) . '</strong><br />';
			echo esc_html( $import_data->get_error_message() ) . '</p>';
			return false;
		}

		$this->version = $import_data['version'];
		if ( $this->version > $this->max_wxr_version ) {
			echo '<div class="error"><p><strong>';
			printf( esc_html__( 'This WXR file (version %s) may not be supported by this version of the importer. Please consider updating.', 'impeka-importer' ), esc_html($import_data['version']) );
			echo '</strong></p></div>';
		}

		$this->get_authors_from_import( $import_data );

		return true;
	}

	/**
	 * Retrieve authors from parsed WXR data
	 *
	 * Uses the provided author information from WXR 1.1 files
	 * or extracts info from each post for WXR 1.0 files
	 *
	 * @param array $import_data Data returned by a WXR parser
	 */
	function get_authors_from_import( $import_data ) {
		if ( ! empty( $import_data['authors'] ) ) {
			$this->authors = $import_data['authors'];
		// no author information, grab it from the posts
		} else {
			foreach ( $import_data['posts'] as $post ) {
				$login = sanitize_user( $post['post_author'], true );
				if ( empty( $login ) ) {
					printf( esc_html__( 'Failed to import author %s. Their posts will be attributed to the current user.', 'impeka-importer' ), esc_html( $post['post_author'] ) );
					echo '<br />';
					continue;
				}

				if ( ! isset($this->authors[$login]) ) {
					$this->authors[$login] = array(
						'author_login' => $login,
						'author_display_name' => $post['post_author']
					);
				}
			}
		}
	}

	/**
	 * Map old author logins to local user IDs based on decisions made
	 * in import options form. Can map to an existing user, create a new user
	 * or falls back to the current user in case of error with either of the previous
	 */
	function get_author_mapping() {
		if ( ! isset( $_POST['imported_authors'] ) )
			return;

		$create_users = $this->allow_create_users();

		foreach ( (array) $_POST['imported_authors'] as $i => $old_login ) {
			// Multisite adds strtolower to sanitize_user. Need to sanitize here to stop breakage in process_posts.
			$santized_old_login = sanitize_user( $old_login, true );
			$old_id = isset( $this->authors[$old_login]['author_id'] ) ? intval($this->authors[$old_login]['author_id']) : false;

			if ( ! empty( $_POST['user_map'][$i] ) ) {
				$user = get_userdata( intval($_POST['user_map'][$i]) );
				if ( isset( $user->ID ) ) {
					if ( $old_id )
						$this->processed_authors[$old_id] = $user->ID;
					$this->author_mapping[$santized_old_login] = $user->ID;
				}
			} else if ( $create_users ) {
				if ( ! empty($_POST['user_new'][$i]) ) {
					$user_id = wp_create_user( $_POST['user_new'][$i], wp_generate_password() );
				} else if ( $this->version != '1.0' ) {
					$user_data = array(
						'user_login' => $old_login,
						'user_pass' => wp_generate_password(),
						'user_email' => isset( $this->authors[$old_login]['author_email'] ) ? $this->authors[$old_login]['author_email'] : '',
						'display_name' => $this->authors[$old_login]['author_display_name'],
						'first_name' => isset( $this->authors[$old_login]['author_first_name'] ) ? $this->authors[$old_login]['author_first_name'] : '',
						'last_name' => isset( $this->authors[$old_login]['author_last_name'] ) ? $this->authors[$old_login]['author_last_name'] : '',
					);
					$user_id = wp_insert_user( $user_data );
				}

				if ( ! is_wp_error( $user_id ) ) {
					if ( $old_id )
						$this->processed_authors[$old_id] = $user_id;
					$this->author_mapping[$santized_old_login] = $user_id;
				} else {
					printf( esc_html__( 'Failed to create new user for %s. Their posts will be attributed to the current user.', 'impeka-importer' ), esc_html($this->authors[$old_login]['author_display_name']) );
					if ( defined('IMPEKA_IMPORT_DEBUG') && IMPEKA_IMPORT_DEBUG ) {
						echo ' ' . $user_id->get_error_message();
					}
					echo '<br />';
				}
			}

			// failsafe: if the user_id was invalid, default to the current user
			if ( ! isset( $this->author_mapping[$santized_old_login] ) ) {
				if ( $old_id )
					$this->processed_authors[$old_id] = (int) get_current_user_id();
				$this->author_mapping[$santized_old_login] = (int) get_current_user_id();
			}
		}
	}

	/**
	 * Create new categories based on import information
	 *
	 * Doesn't create a new category if its slug already exists
	 */
	function process_categories() {
		$this->categories = apply_filters( 'wp_import_categories', $this->categories );

		if ( empty( $this->categories ) )
			return;

		foreach ( $this->categories as $cat ) {
			// if the category already exists leave it alone
			$term_id = term_exists( $cat['category_nicename'], 'category' );
			if ( $term_id ) {
				if ( is_array($term_id) ) $term_id = $term_id['term_id'];
				if ( isset($cat['term_id']) )
					$this->processed_terms[intval($cat['term_id'])] = (int) $term_id;
				continue;
			}

			$parent      = empty( $cat['category_parent'] ) ? 0 : category_exists( $cat['category_parent'] );
			$description = isset( $cat['category_description'] ) ? $cat['category_description'] : '';

			$data = array(
				'category_nicename'    => $cat['category_nicename'],
				'category_parent'      => $parent,
				'cat_name'             => wp_slash( $cat['cat_name'] ),
				'category_description' => wp_slash( $description ),
			);

			$id = wp_insert_category( $data );
			if ( ! is_wp_error( $id ) && $id > 0 ) {
				if ( isset($cat['term_id']) )
					$this->processed_terms[intval($cat['term_id'])] = $id;
			} else {
				if ( defined('IMPEKA_IMPORT_DEBUG') && IMPEKA_IMPORT_DEBUG ) {
					printf( esc_html__( 'Failed to import category %s', 'impeka-importer' ), esc_html($cat['category_nicename']) );
					echo ': ' . $id->get_error_message();
					echo '<br />';
				}
				continue;
			}
			$this->process_termmeta( $cat, $id );
		}

		unset( $this->categories );
	}

	/**
	 * Create new post tags based on import information
	 *
	 * Doesn't create a tag if its slug already exists
	 */
	function process_tags() {
		$this->tags = apply_filters( 'wp_import_tags', $this->tags );

		if ( empty( $this->tags ) )
			return;

		foreach ( $this->tags as $tag ) {
			// if the tag already exists leave it alone
			$term_id = term_exists( $tag['tag_slug'], 'post_tag' );
			if ( $term_id ) {
				if ( is_array($term_id) ) $term_id = $term_id['term_id'];
				if ( isset($tag['term_id']) )
					$this->processed_terms[intval($tag['term_id'])] = (int) $term_id;
				continue;
			}

			$description = isset( $tag['tag_description'] ) ? $tag['tag_description'] : '';
			$args        = array(
				'slug'        => $tag['tag_slug'],
				'description' => wp_slash( $description ),
			);

			$id = wp_insert_term( wp_slash( $tag['tag_name'] ), 'post_tag', $args );
			if ( ! is_wp_error( $id ) ) {
				if ( isset($tag['term_id']) )
					$this->processed_terms[intval($tag['term_id'])] = $id['term_id'];
			} else {
				printf( esc_html__( 'Failed to import post tag %s', 'impeka-importer' ), esc_html($tag['tag_name']) );
				if ( defined('IMPEKA_IMPORT_DEBUG') && IMPEKA_IMPORT_DEBUG )
					echo ': ' . $id->get_error_message();
				echo '<br />';
				continue;
			}

			$this->process_termmeta( $tag, $id['term_id'] );
		}

		unset( $this->tags );
	}

	/**
	 * Create new terms based on import information
	 *
	 * Doesn't create a term its slug already exists
	 */
	function process_terms() {
		$this->terms = apply_filters( 'wp_import_terms', $this->terms );

		if ( empty( $this->terms ) )
			return;

		foreach ( $this->terms as $term ) {
			// if the term already exists in the correct taxonomy leave it alone
			$term_id = term_exists( $term['slug'], $term['term_taxonomy'] );
			if ( $term_id ) {
				if ( is_array($term_id) ) $term_id = $term_id['term_id'];
				if ( isset($term['term_id']) )
					$this->processed_terms[intval($term['term_id'])] = (int) $term_id;
				continue;
			}

			if ( empty( $term['term_parent'] ) ) {
				$parent = 0;
			} else {
				$parent = term_exists( $term['term_parent'], $term['term_taxonomy'] );
				if ( is_array( $parent ) ) {
					$parent = $parent['term_id'];
				}
			}

			$description = isset( $term['term_description'] ) ? $term['term_description'] : '';
			$args        = array(
				'slug'        => $term['slug'],
				'description' => wp_slash( $description ),
				'parent'      => (int) $parent
			);

			$id = wp_insert_term( wp_slash( $term['term_name'] ), $term['term_taxonomy'], $args );
			if ( ! is_wp_error( $id ) ) {
				if ( isset($term['term_id']) )
					$this->processed_terms[intval($term['term_id'])] = $id['term_id'];
			} else {
				if ( defined('IMPEKA_IMPORT_DEBUG') && IMPEKA_IMPORT_DEBUG ) {
					printf( esc_html__( 'Failed to import %s %s', 'impeka-importer' ), esc_html($term['term_taxonomy']), esc_html($term['term_name']) );
					echo ': ' . $id->get_error_message();
					echo '<br />';
				}
				continue;
			}
			$this->process_termmeta( $term, $id['term_id'] );
		}

		unset( $this->terms );
	}

	/**
	 * Add metadata to imported term.
	 *
	 * @param array $term    Term data from WXR import.
	 * @param int   $term_id ID of the newly created term.
	 */
	protected function process_termmeta( $term, $term_id ) {
		if ( ! function_exists( 'add_term_meta' ) ) {
			return;
		}

		if ( ! isset( $term['termmeta'] ) ) {
			$term['termmeta'] = array();
		}

		/**
		 * Filters the metadata attached to an imported term.
		 *
		 * @param array $termmeta Array of term meta.
		 * @param int   $term_id  ID of the newly created term.
		 * @param array $term     Term data from the WXR import.
		 */
		$term['termmeta'] = apply_filters( 'wp_import_term_meta', $term['termmeta'], $term_id, $term );

		if ( empty( $term['termmeta'] ) ) {
			return;
		}

		foreach ( $term['termmeta'] as $meta ) {
			/**
			 * Filters the meta key for an imported piece of term meta.
			 *
			 * @param string $meta_key Meta key.
			 * @param int    $term_id  ID of the newly created term.
			 * @param array  $term     Term data from the WXR import.
			 */
			$key = apply_filters( 'import_term_meta_key', $meta['key'], $term_id, $term );
			if ( ! $key ) {
				continue;
			}

			// Export gets meta straight from the DB so could have a serialized string
			$value = maybe_unserialize( $meta['value'] );

			add_term_meta( $term_id, wp_slash( $key ), wp_slash_strings_only( $value ) );

			/**
			 * Fires after term meta is imported.
			 *
			 * @since 0.6.2
			 *
			 * @param int    $term_id ID of the newly created term.
			 * @param string $key     Meta key.
			 * @param mixed  $value   Meta value.
			 */
			do_action( 'import_term_meta', $term_id, $key, $value );
		}
	}

	/**
	 * Create new posts based on import information
	 *
	 * Posts marked as having a parent which doesn't exist will become top level items.
	 * Doesn't create a new post if: the post type doesn't exist, the given post ID
	 * is already noted as imported or a post with the same title and date already exists.
	 * Note that new/updated terms, comments and meta are imported for the last of the above.
	 */

	function process_posts( $ids_array = array() ) {
		$this->posts = apply_filters( 'wp_import_posts', $this->posts );

		$dummy_image_index = 1;
		foreach ( $this->posts as $post ) {
			$post = apply_filters( 'wp_import_post_data_raw', $post );


			if ( !empty( $ids_array ) && !in_array( $post['post_id'], $ids_array ) ) {
				continue;
			}

			if ( ! post_type_exists( $post['post_type'] ) ) {
				if ( defined('IMPEKA_IMPORT_DEBUG') && IMPEKA_IMPORT_DEBUG ) {
					printf( esc_html__( 'Failed to import &#8220;%s&#8221;: Invalid post type %s', 'impeka-importer' ),
						esc_html($post['post_title']), esc_html($post['post_type']) );
					echo '<br />';
				}
				do_action( 'wp_import_post_exists', $post );
				continue;
			}

			if ( isset( $this->processed_posts[$post['post_id']] ) && ! empty( $post['post_id'] ) )
				continue;

			if ( $post['status'] == 'auto-draft' )
				continue;

			if ( 'nav_menu_item' == $post['post_type'] ) {
				$this->process_menu_item( $post );
				continue;
			}

			$post_type_object = get_post_type_object( $post['post_type'] );
			$post_exists = post_exists( $post['post_title'], '', $post['post_date'] );

			$post_exists = apply_filters( 'wp_import_existing_post', $post_exists, $post );

			if ( $post_exists && get_post_type( $post_exists ) == $post['post_type'] ) {
				if ( defined('IMPEKA_IMPORT_DEBUG') && IMPEKA_IMPORT_DEBUG ) {
					printf( esc_html__('%s &#8220;%s&#8221; already exists.', 'impeka-importer'), $post_type_object->labels->singular_name, esc_html($post['post_title']) );
					echo '<br />';
				}
				$comment_post_ID = $post_id = $post_exists;
				$this->processed_posts[ intval( $post['post_id'] ) ] = intval( $post_exists );
			} else {
				$post_parent = (int) $post['post_parent'];
				if ( $post_parent ) {
					// if we already know the parent, map it to the new local ID
					if ( isset( $this->processed_posts[$post_parent] ) ) {
						$post_parent = $this->processed_posts[$post_parent];
					// otherwise record the parent for later
					} else {
						$this->post_orphans[intval($post['post_id'])] = $post_parent;
						$post_parent = 0;
					}
				}

				// map the post author
				$author = sanitize_user( $post['post_author'], true );
				if ( isset( $this->author_mapping[$author] ) )
					$author = $this->author_mapping[$author];
				else
					$author = (int) get_current_user_id();

				//Remap Categories
				$pattern = '|categories="([^"]*)"|';
				$post['post_content'] = preg_replace_callback($pattern, "self::grve_categories_callback", $post['post_content']);

				if ( 'attachment' != $post['post_type'] ) {
					$post['guid'] = '';
				}
				$postdata = array(
					'import_id' => $post['post_id'], 'post_author' => $author, 'post_date' => $post['post_date'],
					'post_date_gmt' => $post['post_date_gmt'], 'post_content' => $post['post_content'],
					'post_excerpt' => $post['post_excerpt'], 'post_title' => $post['post_title'],
					'post_status' => $post['status'], 'post_name' => $post['post_name'],
					'comment_status' => $post['comment_status'], 'ping_status' => $post['ping_status'],
					'guid' => $post['guid'], 'post_parent' => $post_parent, 'menu_order' => $post['menu_order'],
					'post_type' => $post['post_type'], 'post_password' => $post['post_password']
				);

				$original_post_ID = $post['post_id'];
				$postdata = apply_filters( 'wp_import_post_data_processed', $postdata, $post );
				$postdata = wp_slash( $postdata );
				if ( 'attachment' == $postdata['post_type'] ) {

					$real_remote_url = ! empty($post['attachment_url']) ? $post['attachment_url'] : $post['guid'];
					$remote_url = ! empty($post['attachment_url']) ? $post['attachment_url'] : $post['guid'];

					//Demo Dummy Image
					$parts = pathinfo( $remote_url );
 					if ( ! $this->fetch_live_images && ( 'jpeg' == $parts['extension'] || 'jpg' == $parts['extension'] || 'png' == $parts['extension'] || 'gif' == $parts['extension'] ) ) {
						$postdata['upload_date'] = '2000/01';

						if ( $dummy_image_index > 6 ) {
							$dummy_image_index = 1;
						}
						$remote_url = plugins_url( '/import/dummy/impeka-featured-img-' . $dummy_image_index . '.jpg', __FILE__ );
						$dummy_image_index++;

						if ( isset( $post['postmeta'] ) ) {
							foreach( $post['postmeta'] as $meta ) {
								if ( $meta['key'] == '_wp_attachment_metadata' ) {
									$metavalue = maybe_unserialize( $meta['value'] );
									$width = $metavalue['width'];
									$height = $metavalue['height'];

									if ( $width <= 201 ) {
										$remote_url = plugins_url( '/import/dummy/impeka-featured-img-small.png', __FILE__ );
									}
								}
							}
						}

					} elseif ( 'mp4' == $parts['extension'] || 'webm' == $parts['extension'] || 'ogv' == $parts['extension'] ) {
						continue;
					} elseif ( 'mp3' == $parts['extension'] ) {
						continue;
					} elseif ( 'svg' == $parts['extension'] ) {
						continue;
					} else {

						// try to use _wp_attached file for upload folder placement to ensure the same location as the export site
						// e.g. location is 2003/05/image.jpg but the attachment post_date is 2010/09, see media_handle_upload()
						$postdata['upload_date'] = $post['post_date'];
						if ( isset( $post['postmeta'] ) ) {
							foreach( $post['postmeta'] as $meta ) {
								if ( $meta['key'] == '_wp_attached_file' ) {
									if ( preg_match( '%^[0-9]{4}/[0-9]{2}%', $meta['value'], $matches ) )
										$postdata['upload_date'] = $matches[0];
									break;
								}
							}
						}
 					}

					$comment_post_ID = $post_id = $this->process_attachment( $postdata, $remote_url, $real_remote_url );
				} else {
					$comment_post_ID = $post_id = wp_insert_post( $postdata, true );
					do_action( 'wp_import_insert_post', $post_id, $original_post_ID, $postdata, $post );
				}

				if ( is_wp_error( $post_id ) ) {
					if ( defined('IMPEKA_IMPORT_DEBUG') && IMPEKA_IMPORT_DEBUG ) {
						printf( esc_html__( 'Failed to import %s &#8220;%s&#8221;', 'impeka-importer' ),
						$post_type_object->labels->singular_name, esc_html($post['post_title']) );
						echo ': ' . $post_id->get_error_message();
						echo '<br />';
					}
					continue;
				}

				if ( $post['is_sticky'] == 1 )
					stick_post( $post_id );
			}

			// map pre-import ID to local ID
			$this->processed_posts[intval($post['post_id'])] = (int) $post_id;

			if ( ! isset( $post['terms'] ) )
				$post['terms'] = array();

			$post['terms'] = apply_filters( 'wp_import_post_terms', $post['terms'], $post_id, $post );

			// add categories, tags and other terms
			if ( ! empty( $post['terms'] ) ) {
				$terms_to_set = array();
				foreach ( $post['terms'] as $term ) {
					// back compat with WXR 1.0 map 'tag' to 'post_tag'
					$taxonomy = ( 'tag' == $term['domain'] ) ? 'post_tag' : $term['domain'];
					$term_exists = term_exists( $term['slug'], $taxonomy );
					$term_id = is_array( $term_exists ) ? $term_exists['term_id'] : $term_exists;
					if ( ! $term_id ) {
						$t = wp_insert_term( $term['name'], $taxonomy, array( 'slug' => $term['slug'] ) );
						if ( ! is_wp_error( $t ) ) {
							$term_id = $t['term_id'];
							do_action( 'wp_import_insert_term', $t, $term, $post_id, $post );
						} else {
							if ( defined('IMPEKA_IMPORT_DEBUG') && IMPEKA_IMPORT_DEBUG ) {
								printf( esc_html__( 'Failed to import %s %s', 'impeka-importer' ), esc_html($taxonomy), esc_html($term['name']) );
								echo ': ' . $t->get_error_message();
								echo '<br />';
							}
							do_action( 'wp_import_insert_term_failed', $t, $term, $post_id, $post );
							continue;
						}
					}
					$terms_to_set[$taxonomy][] = intval( $term_id );
				}

				foreach ( $terms_to_set as $tax => $ids ) {
					$tt_ids = wp_set_post_terms( $post_id, $ids, $tax );
					do_action( 'wp_import_set_post_terms', $tt_ids, $ids, $tax, $post_id, $post );
				}
				unset( $post['terms'], $terms_to_set );
			}

			if ( ! isset( $post['comments'] ) )
				$post['comments'] = array();

			$post['comments'] = apply_filters( 'wp_import_post_comments', $post['comments'], $post_id, $post );

			// add/update comments
			if ( ! empty( $post['comments'] ) ) {
				$num_comments = 0;
				$inserted_comments = array();
				foreach ( $post['comments'] as $comment ) {
					$comment_id	= $comment['comment_id'];
					$newcomments[$comment_id]['comment_post_ID']      = $comment_post_ID;
					$newcomments[$comment_id]['comment_author']       = $comment['comment_author'];
					$newcomments[$comment_id]['comment_author_email'] = $comment['comment_author_email'];
					$newcomments[$comment_id]['comment_author_IP']    = $comment['comment_author_IP'];
					$newcomments[$comment_id]['comment_author_url']   = $comment['comment_author_url'];
					$newcomments[$comment_id]['comment_date']         = $comment['comment_date'];
					$newcomments[$comment_id]['comment_date_gmt']     = $comment['comment_date_gmt'];
					$newcomments[$comment_id]['comment_content']      = $comment['comment_content'];
					$newcomments[$comment_id]['comment_approved']     = $comment['comment_approved'];
					$newcomments[$comment_id]['comment_type']         = $comment['comment_type'];
					$newcomments[$comment_id]['comment_parent'] 	  = $comment['comment_parent'];
					$newcomments[$comment_id]['commentmeta']          = isset( $comment['commentmeta'] ) ? $comment['commentmeta'] : array();
					if ( isset( $this->processed_authors[$comment['comment_user_id']] ) )
						$newcomments[$comment_id]['user_id'] = $this->processed_authors[$comment['comment_user_id']];
				}
				ksort( $newcomments );

				foreach ( $newcomments as $key => $comment ) {
					// if this is a new post we can skip the comment_exists() check
					if ( ! $post_exists || ! comment_exists( $comment['comment_author'], $comment['comment_date'] ) ) {
						if ( isset( $inserted_comments[$comment['comment_parent']] ) ) {
							$comment['comment_parent'] = $inserted_comments[ $comment['comment_parent'] ];
						}

						$comment_data = wp_slash( $comment );
						unset( $comment_data['commentmeta'] ); // Handled separately, wp_insert_comment() also expects `comment_meta`.
						$comment_data = wp_filter_comment( $comment_data );

						$inserted_comments[ $key ] = wp_insert_comment( $comment_data );

						do_action( 'wp_import_insert_comment', $inserted_comments[ $key ], $comment, $comment_post_ID, $post );

						foreach( $comment['commentmeta'] as $meta ) {
							$value = maybe_unserialize( $meta['value'] );

							add_comment_meta( $inserted_comments[ $key ], wp_slash( $meta['key'] ), wp_slash_strings_only( $value ) );
						}

						$num_comments++;
					}
				}
				unset( $newcomments, $inserted_comments, $post['comments'] );
			}

			if ( ! isset( $post['postmeta'] ) )
				$post['postmeta'] = array();

			$post['postmeta'] = apply_filters( 'wp_import_post_meta', $post['postmeta'], $post_id, $post );

			// add/update post meta
			if ( ! empty( $post['postmeta'] ) ) {
				foreach ( $post['postmeta'] as $meta ) {
					$key = apply_filters( 'import_post_meta_key', $meta['key'], $post_id, $post );
					$value = false;

					if ( '_edit_last' == $key ) {
						if ( isset( $this->processed_authors[intval($meta['value'])] ) )
							$value = $this->processed_authors[intval($meta['value'])];
						else
							$key = false;
					}

					if ( $key ) {
						// export gets meta straight from the DB so could have a serialized string
						if ( ! $value ) {
							$value = maybe_unserialize( $meta['value'] );
						}

						add_post_meta( $post_id, wp_slash( $key ), wp_slash_strings_only( $value ) );

						do_action( 'import_post_meta', $post_id, $key, $value );

						// if the post has a featured image, take note of this in case of remap
						if ( '_thumbnail_id' == $key )
							$this->featured_images[$post_id] = (int) $value;
					}
				}
			}
		}

		unset( $this->posts );
	}

	/**
	 * Attempt to create a new menu item from import data
	 *
	 * Fails for draft, orphaned menu items and those without an associated nav_menu
	 * or an invalid nav_menu term. If the post type or term object which the menu item
	 * represents doesn't exist then the menu item will not be imported (waits until the
	 * end of the import to retry again before discarding).
	 *
	 * @param array $item Menu item details from WXR file
	 */
	function process_menu_item( $item ) {
		// skip draft, orphaned menu items
		if ( 'draft' == $item['status'] )
			return;

		$menu_slug = false;
		if ( isset($item['terms']) ) {
			// loop through terms, assume first nav_menu term is correct menu
			foreach ( $item['terms'] as $term ) {
				if ( 'nav_menu' == $term['domain'] ) {
					$menu_slug = $term['slug'];
					break;
				}
			}
		}

		// no nav_menu term associated with this menu item
		if ( ! $menu_slug ) {
			if ( defined('IMPEKA_IMPORT_DEBUG') && IMPEKA_IMPORT_DEBUG ) {
				esc_html_e( 'Menu item skipped due to missing menu slug', 'impeka-importer' );
				echo '<br />';
			}
			return;
		}

		$menu_id = term_exists( $menu_slug, 'nav_menu' );
		if ( ! $menu_id ) {
			if ( defined('IMPEKA_IMPORT_DEBUG') && IMPEKA_IMPORT_DEBUG ) {
				printf( esc_html__( 'Menu item skipped due to invalid menu slug: %s', 'impeka-importer' ), esc_html( $menu_slug ) );
				echo '<br />';
			}
			return;
		} else {
			$menu_id = is_array( $menu_id ) ? $menu_id['term_id'] : $menu_id;
		}

		foreach ( $item['postmeta'] as $meta ) {
			if ( version_compare( PHP_VERSION, '7.0.0' ) >= 0 ) {
				${$meta['key']} = $meta['value'];
			} else {
				$$meta['key'] = $meta['value'];
			}
		}

		if ( 'taxonomy' == $_menu_item_type && isset( $this->processed_terms[intval($_menu_item_object_id)] ) ) {
			$_menu_item_object_id = $this->processed_terms[intval($_menu_item_object_id)];
		} else if ( 'post_type' == $_menu_item_type && isset( $this->processed_posts[intval($_menu_item_object_id)] ) ) {
			$_menu_item_object_id = $this->processed_posts[intval($_menu_item_object_id)];
		} else if ( 'custom' != $_menu_item_type ) {
			// associated object is missing or not imported yet, we'll retry later
			$this->missing_menu_items[] = $item;
			return;
		}

		if ( isset( $this->processed_menu_items[intval($_menu_item_menu_item_parent)] ) ) {
			$_menu_item_menu_item_parent = $this->processed_menu_items[intval($_menu_item_menu_item_parent)];
		} else if ( $_menu_item_menu_item_parent ) {
			$this->menu_item_orphans[intval($item['post_id'])] = (int) $_menu_item_menu_item_parent;
			$_menu_item_menu_item_parent = 0;
		}

		// wp_update_nav_menu_item expects CSS classes as a space separated string
		$_menu_item_classes = maybe_unserialize( $_menu_item_classes );
		if ( is_array( $_menu_item_classes ) )
			$_menu_item_classes = implode( ' ', $_menu_item_classes );

		$args = array(
			'menu-item-object-id' => $_menu_item_object_id,
			'menu-item-object' => $_menu_item_object,
			'menu-item-parent-id' => $_menu_item_menu_item_parent,
			'menu-item-position' => intval( $item['menu_order'] ),
			'menu-item-type' => $_menu_item_type,
			'menu-item-title' => $item['post_title'],
			'menu-item-url' => $_menu_item_url,
			'menu-item-description' => $item['post_content'],
			'menu-item-attr-title' => $item['post_excerpt'],
			'menu-item-target' => $_menu_item_target,
			'menu-item-classes' => $_menu_item_classes,
			'menu-item-xfn' => $_menu_item_xfn,
			'menu-item-status' => $item['status']
		);

		$id = wp_update_nav_menu_item( $menu_id, 0, $args );
		if ( $id && ! is_wp_error( $id ) )
			$this->processed_menu_items[intval($item['post_id'])] = (int) $id;

		if ( $id && ! is_wp_error( $id ) ) {
			foreach($item['postmeta'] as $itemkey => $meta) {
				$key = str_replace('_', '-', ltrim($meta['key'], "_"));
				if( !array_key_exists($key, $args) && $key != "menu-item-menu-item-parent") {
					if(!empty($meta['value'])) {
						update_post_meta($id, $meta['key'], $meta['value']);
					}
				}
			}
		}
	}

	/**
	 * If fetching attachments is enabled then attempt to create a new attachment
	 *
	 * @param array $post Attachment post details from WXR
	 * @param string $url URL to fetch attachment from ( Dummy )
	 * @param string $real_url URL to fetch attachment from
	 * @return int|WP_Error Post ID on success, WP_Error otherwise
	 */
	function process_attachment( $post, $url, $real_url ) {
		if ( ! $this->fetch_attachments )
			return new WP_Error( 'attachment_processing_error',
				__( 'Fetching attachments is not enabled', 'impeka-importer' ) );

		// if the URL is absolute, but does not contain address, then upload it assuming base_site_url
		if ( preg_match( '|^/[\w\W]+$|', $url ) )
			$url = rtrim( $this->base_url, '/' ) . $url;

		if ( preg_match( '|^/[\w\W]+$|', $real_url ) )
			$real_url = rtrim( $this->base_url, '/' ) . $real_url;

		$upload = $this->fetch_remote_file( $url, $real_url, $post );

		if ( is_wp_error( $upload ) ) {
			return $upload;
		}
		if ( $info = wp_check_filetype( $upload['file'] ) )
			$post['post_mime_type'] = $info['type'];
		else
			return new WP_Error( 'attachment_processing_error', esc_html__('Invalid file type', 'impeka-importer') );

		$post['guid'] = $upload['url'];

		// as per wp-admin/includes/upload.php
		$post_id = wp_insert_attachment( $post, $upload['file'] );
		if ( isset( $upload['metadata'] ) && !empty( $upload['metadata'] ) ) {
			wp_update_attachment_metadata( $post_id, $upload['metadata'] );
		} else {
			wp_update_attachment_metadata( $post_id, wp_generate_attachment_metadata( $post_id, $upload['file'] ) );
		}

		// remap resized image URLs, works by stripping the extension and remapping the URL stub.
		if ( preg_match( '!^image/!', $info['type'] ) ) {
			$parts = pathinfo( $url );
			$name = basename( $parts['basename'], ".{$parts['extension']}" ); // PATHINFO_FILENAME in PHP 5.2

			$parts_new = pathinfo( $upload['url'] );
			$name_new = basename( $parts_new['basename'], ".{$parts_new['extension']}" );

			$this->url_remap[$parts['dirname'] . '/' . $name] = $parts_new['dirname'] . '/' . $name_new;
		}

		return $post_id;
	}

	/**
	 * Attempt to download a remote file attachment
	 *
	 * @param string $url URL of item to fetch
	 * @param array $post Attachment details
	 * @return array|WP_Error Local file location details on success, WP_Error otherwise
	 */
	function fetch_remote_file( $url, $real_url, $post ) {
		// Extract the file name from the URL.
		$file_name = basename( parse_url( $url, PHP_URL_PATH ) );

		//Check if attachment already uploaded
		$upload_dir = wp_upload_dir( $post['upload_date'] );
		$upload_file_dir = $upload_dir['path'] . '/' . $file_name;
		if( file_exists( $upload_file_dir ) && ! $this->fetch_live_images ) {
			$upload_file_url = $upload_dir['url'] . '/' . $file_name;
			$import_file = plugin_dir_path(__FILE__) . 'import/dummy/' . $file_name . '.json' ;
			$meta_array = array();
			if ( file_exists( $import_file ) ) {
				$import_array = file_get_contents( $import_file );
				$meta_array = json_decode( $import_array, true );
			}

			$upload = array(
				'file' => $upload_file_dir,
				'url' => $upload_file_url,
				'metadata' => $meta_array,
				'error' => false,
			);
		} else {
			// get placeholder file in the upload dir with a unique, sanitized filename
			$upload = wp_upload_bits( $file_name, 0, '', $post['upload_date'] );

			if ( $upload['error'] )
				return new WP_Error( 'upload_dir_error', $upload['error'] );

			// fetch the remote url and write it to the placeholder file
			$remote_response = wp_safe_remote_get( $url, array(
				'timeout' => 300,
						'stream' => true,
						'filename' => $upload['file'],
				) );

			$headers = wp_remote_retrieve_headers( $remote_response );

			// request failed
			if ( ! $headers ) {
				@unlink( $upload['file'] );
				return new WP_Error( 'import_file_error', esc_html__('Remote server did not respond', 'impeka-importer') );
			}

			$remote_response_code = wp_remote_retrieve_response_code( $remote_response );

			// make sure the fetch was successful
			if ( $remote_response_code != '200' ) {
				@unlink( $upload['file'] );
				return new WP_Error( 'import_file_error', sprintf( esc_html__('Remote server returned error response %1$d %2$s', 'impeka-importer'), esc_html($remote_response_code), get_status_header_desc($remote_response_code) ) );
			}

			$filesize = filesize( $upload['file'] );

			if ( 0 == $filesize ) {
				@unlink( $upload['file'] );
				return new WP_Error( 'import_file_error', esc_html__('Zero size file downloaded', 'impeka-importer') );
			}

			$max_size = (int) $this->max_attachment_size();
			if ( ! empty( $max_size ) && $filesize > $max_size ) {
				@unlink( $upload['file'] );
				return new WP_Error( 'import_file_error', sprintf(__('Remote file is too large, limit is %s', 'impeka-importer'), size_format($max_size) ) );
			}
		}

		// keep track of the old and new urls so we can substitute them later
		$this->url_remap[$real_url] = $upload['url'];
		// keep track of the destination if the remote url is redirected somewhere else
		if ( isset($headers['x-final-location']) && $headers['x-final-location'] != $url )
			$this->url_remap[$headers['x-final-location']] = $upload['url'];

		return $upload;
	}

	/**
	 * Attempt to associate posts and menu items with previously missing parents
	 *
	 * An imported post's parent may not have been imported when it was first created
	 * so try again. Similarly for child menu items and menu items which were missing
	 * the object (e.g. post) they represent in the menu
	 */
	function backfill_parents() {
		global $wpdb;

		// find parents for post orphans
		foreach ( $this->post_orphans as $child_id => $parent_id ) {
			$local_child_id = $local_parent_id = false;
			if ( isset( $this->processed_posts[$child_id] ) )
				$local_child_id = $this->processed_posts[$child_id];
			if ( isset( $this->processed_posts[$parent_id] ) )
				$local_parent_id = $this->processed_posts[$parent_id];

			if ( $local_child_id && $local_parent_id ) {
				$wpdb->update( $wpdb->posts, array( 'post_parent' => $local_parent_id ), array( 'ID' => $local_child_id ), '%d', '%d' );
				clean_post_cache( $local_child_id );
			}
		}

		// all other posts/terms are imported, retry menu items with missing associated object
		$missing_menu_items = $this->missing_menu_items;
		foreach ( $missing_menu_items as $item )
			$this->process_menu_item( $item );

		// find parents for menu item orphans
		foreach ( $this->menu_item_orphans as $child_id => $parent_id ) {
			$local_child_id = $local_parent_id = 0;
			if ( isset( $this->processed_menu_items[$child_id] ) )
				$local_child_id = $this->processed_menu_items[$child_id];
			if ( isset( $this->processed_menu_items[$parent_id] ) )
				$local_parent_id = $this->processed_menu_items[$parent_id];

			if ( $local_child_id && $local_parent_id )
				update_post_meta( $local_child_id, '_menu_item_menu_item_parent', (int) $local_parent_id );
		}
	}

	/**
	 * Use stored mapping information to update old attachment URLs
	 */
	function backfill_attachment_urls() {
		global $wpdb;
		// make sure we do the longest urls first, in case one is a substring of another
		uksort( $this->url_remap, array(&$this, 'cmpr_strlen') );

		foreach ( $this->url_remap as $from_url => $to_url ) {
			// remap urls in post_content
			$wpdb->query( $wpdb->prepare("UPDATE {$wpdb->posts} SET post_content = REPLACE(post_content, %s, %s)", $from_url, $to_url) );
			// remap _wpb_shortcodes_custom_css urls
			$result = $wpdb->query( $wpdb->prepare("UPDATE {$wpdb->postmeta} SET meta_value = REPLACE(meta_value, %s, %s) WHERE meta_key='_wpb_shortcodes_custom_css'", $from_url, $to_url) );
			// remap enclosure urls
			$result = $wpdb->query( $wpdb->prepare("UPDATE {$wpdb->postmeta} SET meta_value = REPLACE(meta_value, %s, %s) WHERE meta_key='enclosure'", $from_url, $to_url) );
		}
	}

	/**
	 * Update _thumbnail_id meta to new, imported attachment IDs
	 */
	function remap_featured_images() {
		// cycle through posts that have a featured image
		foreach ( $this->featured_images as $post_id => $value ) {
			if ( isset( $this->processed_posts[$value] ) ) {
				$new_id = $this->processed_posts[$value];
				// only update if there's a difference
				if ( $new_id != $value )
					update_post_meta( $post_id, '_thumbnail_id', $new_id );
			}
		}
	}

	/**
	 * Parse a WXR file
	 *
	 * @param string $file Path to WXR file for parsing
	 * @return array Information gathered from the WXR file
	 */
	function parse( $file ) {
		$parser = new IMPEKA_GRVE_WXR_Parser();
		return $parser->parse( $file );
	}

	// Display import page title
	function header() {
		echo '<div class="wrap">';
	}

	// Close div.wrap
	function footer() {
		echo '</div>';
	}


	/**
	 * Display demo selection
	 */
	function demo_theme_selection() {

		$grve_dummy_data_list = $this->grve_get_demo_data();
?>
		<div class="grve-admin-panel-content">
			<div class="grve-demos">
<?php
		foreach ( $grve_dummy_data_list as $grve_dummy_data ) {
			$image_src = plugins_url( '/import/data/' . $grve_dummy_data['dir'] . '/screenshot.jpg', __FILE__ );
			if ( !$this->is_theme_registered() ) {
				$link = admin_url( 'admin.php?page=impeka' );
				$theme_status = 'unregister';
			} else {
				$link = admin_url( 'admin.php?import=impeka-demo-importer&amp;step=2&amp;demo=' . esc_attr( $grve_dummy_data['dir'] ) );
				$theme_status = 'register';
			}
?>
				<div class="grve-demo-item">
					<div class="grve-demo-item-inner">
						<a href="<?php echo esc_url( $link  ); ?>">
							<div class="grve-demo-media grve-<?php echo esc_attr( $theme_status ); ?>">
								<div class="grve-demo-button">
									<?php if ( !$this->is_theme_registered() ) { ?>
									<svg width="16px" height="16px" viewBox="0 0 16 16"><path d="M8,0 C10.2319008,0 12.0474116,1.84370587 12.1493331,4.15190158 L12.1538462,4.35673624 L12.153,7.772 L14,7.77229602 L14,16 L2,16 L2,7.77229602 L3.846,7.772 L3.84615385,4.35673624 C3.84615385,1.95375405 5.70245502,0 8,0 Z M13.0769231,8.71341176 L2.92307692,8.71341176 L2.92307692,15.0588235 L13.0769231,15.0588235 L13.0769231,8.71341176 Z M8,10.2011385 C8.76803859,10.2011385 9.38461538,10.8500846 9.38461538,11.6432638 C9.38461538,12.2617661 9.00970513,12.7925674 8.48041608,12.996361 L8.47963801,13.5863378 L7.53846154,13.5863378 L7.53746089,13.003098 C6.99875653,12.8044696 6.61538462,12.2687069 6.61538462,11.6432638 C6.61538462,10.8500846 7.23196141,10.2011385 8,10.2011385 Z M8,11.142315 C7.74843598,11.142315 7.53846154,11.3633128 7.53846154,11.6432638 C7.53846154,11.9232148 7.74843598,12.1442125 8,12.1442125 C8.25156402,12.1442125 8.46153846,11.9232148 8.46153846,11.6432638 C8.46153846,11.3633128 8.25156402,11.142315 8,11.142315 Z M8,0.941176471 C6.27638348,0.941176471 4.86310455,2.37013084 4.77371882,4.17503197 L4.76923077,4.35673624 L4.769,7.772 L11.23,7.772 L11.2307692,4.35673624 C11.2307692,2.46698219 9.78107041,0.941176471 8,0.941176471 Z"></path></svg>
									<span><?php esc_html_e( "Register to import", 'impeka-importer' ); ?></span>
									<?php } else { ?>
									<svg width="16px" height="16px" viewBox="0 0 16 16"><path d="M8.47058824,7.15905768 L8.47,13.555 L9.61564096,12.4502622 L10.270507,13.123979 L8.32743304,15 L7.67256696,15 L5.72949295,13.123979 L6.38435904,12.4502622 L7.529,13.555 L7.52941176,7.15905768 L8.47058824,7.15905768 Z M16,2 L16,11.3801049 L9.943074,11.3801049 L9.943074,10.4420944 L15.0588235,10.4420944 L15.0588235,2.93801049 L0.941176471,2.93801049 L0.941176471,10.4420944 L6.056926,10.4420944 L6.056926,11.3801049 L0,11.3801049 L0,2 L16,2 Z"></path></svg>
									<span><?php esc_html_e( "Select import options", 'impeka-importer' ); ?></span>
									<?php } ?>
								</div>
								<img src="<?php echo esc_url( $image_src ); ?>" alt="<?php esc_attr( $grve_dummy_data['name'] ); ?>"/>
							</div>
							<h3 class="grve-demo-title"><?php echo esc_html( $grve_dummy_data['name'] ); ?></h3>
							<p><?php echo esc_html( $grve_dummy_data['description'] ); ?></p>
							<?php $this->grve_print_featured_plugins( $grve_dummy_data['featured_plugins'] ); ?>
						</a>
					</div>
				</div>
<?php
		}
?>
			</div>
		</div>
<?php
	}

	/**
	 * Display introductory text
	 */
	function steps( $step = 1 ) {
?>
		<!-- Steps -->
		<ul class="grve-steps">
			<li class="<?php if ( $step >= 1 ) { echo 'active'; } ?>"><div class="grve-step-number">1</div><span><?php esc_html_e( 'Demos', 'impeka-importer' ); ?></span></li>
			<li class="<?php if ( $step >= 2 ) { echo 'active'; } ?>"><div class="grve-step-number">2</div><span><?php esc_html_e( 'Content', 'impeka-importer' ); ?></span></li>
			<li class="<?php if ( $step >= 3 ) { echo 'active'; } ?>"><div class="grve-step-number">3</div><span><?php esc_html_e( 'Finish', 'impeka-importer' ); ?></span></li>
		</ul>
		<!-- End Steps -->
<?php
	}

	/**
	 * Display introductory text
	 */
	function greet() {
?>
	<div id="grve-importer-wrapper">

		<!-- Welcome -->
		<div id="grve-importer-welcome-page" class="grve-content-wrapper">
			<!-- Content -->
			<div class="grve-importer-content">
				<div class="grve-importer-intro-title">
					<h2 class="grve-title"><?php esc_html_e( "Welcome to Impeka Importer", 'impeka' ); ?></h2>
				</div>
				<div class="grve-importer-content">
				<?php
					if ( !$this->is_theme_active() ) {
				?>
					<div class="grve-importer-notice-required">
						<strong><?php esc_html_e( 'Theme is not active! ', 'impeka-importer' ); ?></strong>
						<br><br>
						<?php echo esc_html__( "In order to import theme demos it is required to install and activate Impeka theme.", 'impeka-importer' ); ?>
					</div>
				<?php
					} elseif ( !$this->is_theme_registered() ) {
				?>
					<div class="grve-importer-notice-required">
						<strong><?php esc_html_e( 'Theme is not registered! ', 'impeka-importer' ); ?></strong>
						<br><br>
						<?php echo sprintf( esc_html__( "In order to import theme demos it is required to %s.", 'impeka' ), '<a href="' . esc_url( admin_url()  . 'admin.php?page=impeka' ) . '" >' . esc_html__( 'register your product', 'impeka-importer' ) . '</a>' ); ?>
					</div>
				<?php
					} else {
				?>
					<div class="grve-buttons-wrapper">
						<a href="<?php echo admin_url( 'admin.php?page=impeka-import' ); ?>" class="grve-btn grve-btn-primary"><?php echo esc_html__( 'Choose Demos', 'impeka-importer' ); ?></a>
					</div>
				<?php
					}
				?>
			</div>
		</div>
	</div>
	<!-- End Importer Wrapper -->
<?php

	}

	/**
	 * Display content selection
	 */
	function contentSelection() {

	$dummy_id = 'undefined';
	if ( isset( $_GET['demo'] ) && !empty( $_GET['demo'] ) ) {
		$dummy_id = $_GET['demo'];
	}
	$grve_dummy_data = array(
		'id'   => 'undefined',
		'name' => 'Undefined Demo',
		'dir'  => 'undefined',
		'description'  => esc_html( 'Undefined Demo', 'impeka-importer' ),
	);

	$grve_dummy_data_list = $this->grve_get_demo_data();
	foreach ( $grve_dummy_data_list as $grve_dummy_data_item ) {
		if( $dummy_id === $grve_dummy_data_item['dir'] ) {
			$grve_dummy_data = $grve_dummy_data_item;
			break;
		}
	}

?>
	<div id="grve-importer-wrapper">
		<?php $this->steps(2); ?>


		<?php
			$inactive_plugins = $this->grve_get_inactive_plugins( $grve_dummy_data['required_plugins'] );
			$import_url = admin_url( 'admin.php?page=impeka-tgmpa-install-plugins' );

			if ( !$this->is_theme_active() ) {
		?>
		<div id="grve-importer-contents-page" class="grve-content-wrapper">
			<div class="grve-importer-content grve-admin-dummy-item">
				<div class="grve-importer-content">
					<div class="grve-importer-intro-title">
						<h2 class="grve-title"><span><?php echo esc_html( $grve_dummy_data['name'] ); ?></span></h2>
					</div>
				</div>
				<div class="grve-importer-notice-required">
					<strong><?php esc_html_e( 'Theme is not active! ', 'impeka-importer' ); ?></strong>
					<br><br>
					<?php echo esc_html__( "In order to import this demo it is required to install and activate Impeka theme.", 'impeka-importer' ); ?>
				</div>
			</div>
		</div>
		<?php
			} elseif ( !$this->is_theme_registered() ) {
		?>
		<div id="grve-importer-contents-page" class="grve-content-wrapper">
			<div class="grve-importer-content grve-admin-dummy-item">
				<div class="grve-importer-content">
					<div class="grve-importer-intro-title">
						<h2 class="grve-title"><span><?php echo esc_html( $grve_dummy_data['name'] ); ?></span></h2>
					</div>
				</div>
				<div class="grve-importer-notice-required">
					<strong><?php esc_html_e( 'Theme is not registered! ', 'impeka-importer' ); ?></strong>
					<br><br>
					<?php echo sprintf( esc_html__( "In order to import this demo it is required to %s.", 'impeka' ), '<a href="' . esc_url( admin_url()  . 'admin.php?page=impeka' ) . '" >' . esc_html__( 'register your product', 'impeka-importer' ) . '</a>' ); ?>
				</div>
			</div>
		</div>
		<?php
			} else if ( !empty( $inactive_plugins ) ) {
		?>
		<div id="grve-importer-contents-page" class="grve-content-wrapper">
			<div class="grve-importer-content grve-admin-dummy-item">
				<div class="grve-importer-content">
					<div class="grve-importer-intro-title">
						<h2 class="grve-title"><span><?php echo esc_html( $grve_dummy_data['name'] ); ?></span></h2>
					</div>
				</div>
				<div class="grve-importer-notice-required">
					<?php esc_html_e( 'Missing plugin(s)', 'impeka-importer' ); ?>: <strong><?php echo implode( ', ', $inactive_plugins ); ?></strong>
					<br><br>
					<?php esc_html_e( 'In order to install this demo it is required to install/activate the missing plugin(s).', 'impeka-importer' ); ?>
					<a href="<?php echo esc_url( $import_url ); ?>"><?php esc_html_e( "Install/Activate Missing Plugins", 'impeka-importer' ); ?></a>
				</div>
			</div>
		</div>
		<?php
			} else {
		?>

		<!-- Installation Loader -->
		<div id="grve-importer-loader" class="grve-content-wrapper" style="display:none;">
			<div class="grve-importer-content">
				<div id="grve-import-loading">
					<div class="grve-spinner"></div>
					<h3 class="grve-importer-title grve-align-center"><?php esc_html_e( 'Import in progress...', 'impeka-importer' ); ?></h3>
					<strong class="grve-align-center"><?php esc_html_e( 'Don\'t close the browser or navigate away', 'impeka-importer' ); ?></strong>
				</div>
				<span id="grve-import-countdown" class="clearfix"></span>
				<div class="clear"></div>
				<div id="grve-import-output-info" class="wrap clearfix" style="display:none;"></div>
				<div id="grve-import-output-container" style="display:none;"></div>
			</div>
		</div>
		<!-- End Installation -->

		<!-- Contents -->
		<div id="grve-importer-contents-page" class="grve-content-wrapper">
			<?php
				$check_widget_selector = isset( $grve_dummy_data['widget_selector'] ) ? $grve_dummy_data['widget_selector'] : false;
				$check_single_selector = isset( $grve_dummy_data['single_selector'] ) ? $grve_dummy_data['single_selector'] : false;
				if ( $check_single_selector ) {
			?>
			<div class="grve-importer-content grve-admin-dummy-item">
				<div class="grve-importer-content">
					<div class="grve-importer-intro-title">
						<h2 class="grve-title"><span><?php echo esc_html( $grve_dummy_data['name'] ); ?></span><?php esc_html_e( 'Import On Demand', 'impeka-importer' ); ?></h2>
						<p class="grve-description"><?php esc_html_e( 'Choose the specific pages, portfolios, posts and products you wish to import into your site. Select the ones you prefer via the available multi selectors below and click the button Import Selected.', 'impeka-importer' ); ?></p>
					</div>
				</div>
				<input type="hidden" class="grve-admin-dummy-option-singular" value="true"/>
				<input type="hidden" class="grve-admin-dummy-option-dummy-nonce" value="<?php echo wp_create_nonce( $grve_dummy_data['dir'] ); ?>"/>
				<?php $this->grve_print_singular_selectors( $grve_dummy_data['dir'] ); ?>
				<div class="grve-importer-info">
					<?php esc_html_e( 'You can select/deselect multiple items too.', 'impeka-importer' ); ?><br><?php esc_html_e( '(ctrl + click on PC and command + click on Mac).', 'impeka-importer' ); ?>
				</div>
				<hr>
				<div class="grve-checkbox"><input type="checkbox" class="grve-admin-dummy-option-single-demo-images" value="yes"/><?php esc_html_e( 'Download images from live demo *', 'impeka-importer' ); ?></div>
				<br><br>
				<div class="grve-importer-small-info">
					<strong>
					* <?php esc_html_e( 'Important Note: ', 'impeka-importer' ); ?>
					</strong>
					<?php esc_html_e( 'The actual images included in preview are for demonstration purposes only. They have been purchased from Shutterstock and/or downloaded from Freepik, Unsplash, Death to Stock and similar sources. In case you use any of these images in your end-product, ensure that they are properly licensed. Greatives does not transfer any licenses for the images used in the demo sites.', 'impeka-importer' ); ?>
				</div>
				<div class="grve-buttons-wrapper">
					<a href="#" class="grve-btn grve-btn-grey grve-import-clear-selection"><?php echo esc_html__( 'Clear Selection', 'impeka-importer' ); ?></a>
					<a href="#" class="grve-btn grve-btn-primary grve-import-dummy-data" data-dummy-id="<?php echo $grve_dummy_data['dir']; ?>"><?php echo esc_html__( 'Import Selected', 'impeka-importer' ); ?></a>
				</div>
			</div>
			<?php
				}
			?>
			<div class="grve-importer-content grve-admin-dummy-item">
				<div class="grve-importer-content">
					<div class="grve-importer-intro-title">
						<h2 class="grve-title"><span><?php echo esc_html( $grve_dummy_data['name'] ); ?></span><?php esc_html_e( 'Import Full Demo', 'impeka-importer' ); ?></h2>
						<p class="grve-description"><?php esc_html_e( 'The import process will work best on a clean installation. You can use a plugin such as WordPress Reset to clear your data first.', 'impeka-importer' ); ?></p>
					</div>
				</div>
				<input type="hidden" class="grve-admin-dummy-option-dummy-nonce" value="<?php echo wp_create_nonce( $grve_dummy_data['dir'] ); ?>"/>
				<div class="grve-admin-dummy-option-wrapper">
					<div class="grve-admin-dummy-option">
						<div class="grve-checkbox"><input type="checkbox" class="grve-admin-dummy-option-dummy-content" value="yes"/><?php esc_html_e( 'Dummy Content', 'impeka-importer' ); ?></div>
						<div class="grve-checkbox"><input type="checkbox" class="grve-admin-dummy-option-theme-options" value="yes"/><?php esc_html_e( 'Theme Options', 'impeka-importer' ); ?></div>
						<?php if ( $check_widget_selector ) { ?>
						<div class="grve-checkbox"><input type="checkbox" class="grve-admin-dummy-option-widgets" value="yes"/><?php esc_html_e( 'Widgets', 'impeka-importer' ); ?></div>
						<?php } ?>
						<div class="grve-checkbox"><input type="checkbox" class="grve-admin-dummy-option-full-demo-images" value="yes"/><?php esc_html_e( 'Download images from live demo *', 'impeka-importer' ); ?></div>
					</div>
					<br><br>
					<div class="grve-importer-small-info">
						<strong>
						* <?php esc_html_e( 'Important Note: ', 'impeka-importer' ); ?>
						</strong>
						<?php esc_html_e( 'The actual images included in preview are for demonstration purposes only. They have been purchased from Shutterstock and/or downloaded from Freepik, Unsplash, Death to Stock and similar sources. In case you use any of these images in your end-product, ensure that they are properly licensed. Greatives does not transfer any licenses for the images used in the demo sites.', 'impeka-importer' ); ?>
					</div>
				</div>
				<div class="grve-buttons-wrapper">
					<a href="#" class="grve-btn grve-btn-primary grve-import-dummy-data" data-dummy-id="<?php echo $grve_dummy_data['dir']; ?>"><?php echo esc_html__( 'Import Full Demo', 'impeka-importer' ); ?></a>
				</div>
			</div>
		</div>
		<!-- End Contents -->

		<form id="grve-import-finish-form" action="<?php echo admin_url( 'admin.php?import=impeka-demo-importer&amp;step=3' ); ?>" method="post"></form>

		<?php
			}
		?>
	</div>
	<!-- End Importer Wrapper -->
<?php

	}
	/**
	 * Display finish page
	 */
	function finish() {

?>
	<div id="grve-importer-wrapper">
		<?php $this->steps(3); ?>
		<!-- Finish -->
		<div id="grve-importer-finish-page" class="grve-content-wrapper">
			<div class="grve-importer-content">
				<div class="grve-completed-icon dashicons dashicons-yes grve-green"></div>
				<h3 class="grve-importer-title grve-align-center"><?php esc_html_e( 'Import', 'impeka-importer' ); ?><br><?php esc_html_e( 'Completed!', 'impeka-importer' ); ?></h3>
				<div class="grve-buttons-wrapper grve-align-center">
					<a href="<?php echo esc_url( home_url() ); ?>" class="grve-btn grve-btn-primary" target="_blank"><?php echo esc_html__( 'Visit Your Site', 'impeka-importer' ); ?></a>
				</div>
			</div>
		</div>
		<!-- End Finish -->
	</div>
	<!-- End Importer Wrapper -->
<?php
	}

	/**
	 * Decide if the given meta key maps to information we will want to import
	 *
	 * @param string $key The meta key to check
	 * @return string|bool The key if we do want to import, false if not
	 */
	function is_valid_meta_key( $key ) {
		// skip attachment metadata since we'll regenerate it from scratch
		// skip _edit_lock as not relevant for import
		if ( in_array( $key, array( '_wp_attached_file', '_wp_attachment_metadata', '_edit_lock' ) ) )
			return false;
		return $key;
	}

	/**
	 * Decide whether or not the importer is allowed to create users.
	 * Default is true, can be filtered via import_allow_create_users
	 *
	 * @return bool True if creating users is allowed
	 */
	function allow_create_users() {
		return apply_filters( 'import_allow_create_users', true );
	}

	/**
	 * Decide whether or not the importer should attempt to download attachment files.
	 * Default is true, can be filtered via import_allow_fetch_attachments. The choice
	 * made at the import options screen must also be true, false here hides that checkbox.
	 *
	 * @return bool True if downloading attachments is allowed
	 */
	function allow_fetch_attachments() {
		return apply_filters( 'import_allow_fetch_attachments', true );
	}

	/**
	 * Decide what the maximum file size for downloaded attachments is.
	 * Default is 0 (unlimited), can be filtered via import_attachment_size_limit
	 *
	 * @return int Maximum attachment file size to import
	 */
	function max_attachment_size() {
		return apply_filters( 'import_attachment_size_limit', 0 );
	}

	/**
	 * Added to http_request_timeout filter to force timeout at 180 seconds during import
	 * @return int 180
	 */
	function bump_request_timeout( $val ) {
		return 180;
	}

	// return the difference in length between two strings
	function cmpr_strlen( $a, $b ) {
		return strlen($b) - strlen($a);
	}

	/**
	 * Parses filename from a Content-Disposition header value.
	 *
	 * As per RFC6266:
	 *
	 *     content-disposition = "Content-Disposition" ":"
	 *                            disposition-type *( ";" disposition-parm )
	 *
	 *     disposition-type    = "inline" | "attachment" | disp-ext-type
	 *                         ; case-insensitive
	 *     disp-ext-type       = token
	 *
	 *     disposition-parm    = filename-parm | disp-ext-parm
	 *
	 *     filename-parm       = "filename" "=" value
	 *                         | "filename*" "=" ext-value
	 *
	 *     disp-ext-parm       = token "=" value
	 *                         | ext-token "=" ext-value
	 *     ext-token           = <the characters in token, followed by "*">
	 *
	 * @since 0.7.0
	 *
	 * @see WP_REST_Attachments_Controller::get_filename_from_disposition()
	 *
	 * @link http://tools.ietf.org/html/rfc2388
	 * @link http://tools.ietf.org/html/rfc6266
	 *
	 * @param string[] $disposition_header List of Content-Disposition header values.
	 * @return string|null Filename if available, or null if not found.
	 */
	protected static function get_filename_from_disposition( $disposition_header ) {
		// Get the filename.
		$filename = null;

		foreach ( $disposition_header as $value ) {
			$value = trim( $value );

			if ( strpos( $value, ';' ) === false ) {
				continue;
			}

			list( $type, $attr_parts ) = explode( ';', $value, 2 );

			$attr_parts = explode( ';', $attr_parts );
			$attributes = array();

			foreach ( $attr_parts as $part ) {
				if ( strpos( $part, '=' ) === false ) {
					continue;
				}

				list( $key, $value ) = explode( '=', $part, 2 );

				$attributes[ trim( $key ) ] = trim( $value );
			}

			if ( empty( $attributes['filename'] ) ) {
				continue;
			}

			$filename = trim( $attributes['filename'] );

			// Unquote quoted filename, but after trimming.
			if ( substr( $filename, 0, 1 ) === '"' && substr( $filename, -1, 1 ) === '"' ) {
				$filename = substr( $filename, 1, -1 );
			}
		}

		return $filename;
	}

	/**
	 * Retrieves file extension by mime type.
	 *
	 * @since 0.7.0
	 *
	 * @param string $mime_type Mime type to search extension for.
	 * @return string|null File extension if available, or null if not found.
	 */
	protected static function get_file_extension_by_mime_type( $mime_type ) {
		static $map = null;

		if ( is_array( $map ) ) {
			return isset( $map[ $mime_type ] ) ? $map[ $mime_type ] : null;
		}

		$mime_types = wp_get_mime_types();
		$map        = array_flip( $mime_types );

		// Some types have multiple extensions, use only the first one.
		foreach ( $map as $type => $extensions ) {
			$map[ $type ] = strtok( $extensions, '|' );
		}

		return isset( $map[ $mime_type ] ) ? $map[ $mime_type ] : null;
	}

	//IMEKA IMPORTER FUNCTIONS START

	/**
	 * Imports stylesheet and scripts
	 */
	function impeka_demo_importer_scripts( $hook ) {

		if ( 'admin.php' == $hook ) {

			wp_register_style( 'grve-import', plugins_url( '/assets/css/grve-import.css', __FILE__  ), array(), '2.5' );
			wp_register_style( 'grve-import-countdown', plugins_url( '/assets/css/jquery.countdown.css', __FILE__  ), array(), '1.0' );

			$grve_import_texts = array(
				'ajaxurl' => admin_url( 'admin-ajax.php' ),
				'split_enabled' => IMPEKA_IMPORT_SPLIT_ATTACHMENTS,
				'debug_enabled' => IMPEKA_IMPORT_DEBUG,
				'confirmation_text' => esc_html__( "You are about to import dummy data!\nIt is recommended to clear your data first.\n\nDo you want to continue?", 'impeka-importer' ),
				'confirmation_text_singular' => esc_html__( "You are about to import partial dummy data!\n\nDo you want to continue?", 'impeka-importer' ),
				'error_text' => esc_html__( "An unexpected error occured:", 'impeka-importer' ),
			);

			wp_register_script( 'grve-import-script', plugins_url( '/assets/js/grve-import.js', __FILE__  ), array( 'jquery'), time(), true );
			wp_localize_script( 'grve-import-script', 'grve_import_texts', $grve_import_texts );
			wp_register_script( 'grve-import-plugin-script', plugins_url( '/assets/js/jquery.plugin.min.js', __FILE__  ), array( 'jquery'), '2.0', true );
			wp_register_script( 'grve-import-countdown-script', plugins_url( '/assets/js/jquery.countdown.min.js', __FILE__  ), array( 'jquery'), '1.0', true );

			wp_enqueue_style( 'grve-import' );
			wp_enqueue_style( 'grve-import-countdown' );
			wp_enqueue_script( 'grve-import-script' );
			wp_enqueue_script( 'grve-import-plugin-script' );
			wp_enqueue_script( 'grve-import-countdown-script' );
		}
	}

	/**
	 * Remap Categories in shortcodes
	 *
	 */
	function grve_categories_callback( $matches ) {
		$matches[0] = '';
		foreach( $matches as $match ){
			$cats = explode(",", $match);
			$new_cats = array();
			foreach( $cats as $cat ){
				if ( isset( $this->processed_terms[intval($cat)] ) ) {
					array_push($new_cats, $this->processed_terms[intval($cat)]);
				}
			}
			if(!empty($new_cats)){
				$matches[0] .= 'categories="' . implode(",", $new_cats) . '"';
			}
		}
		return $matches[0];
	}


	/**
	 * Print Singular Selectors
	 */
	function grve_print_singular_selectors( $dummy_id ) {

		$file = plugin_dir_path(__FILE__) . 'import/data/' . $dummy_id  .  '/dummy.xml';
		if ( ! file_exists( $file ) ) {
			$file = plugin_dir_path(__FILE__) . 'import/data/' . $dummy_id  .  '/dummy.xml.gz';
		}

		if ( ! is_file($file) ) {
			echo '<p><strong>' . esc_html__( 'Sorry, there has been an error.', 'impeka-importer' ) . '</strong><br />';
			echo esc_html__( 'The file/demo does not exist, please try again.', 'impeka-importer' ) . '</p>';
			$this->footer();
			die();
		}

		$import_data = $this->parse( $file );

		if ( is_wp_error( $import_data ) ) {
			echo '<p><strong>' . esc_html__( 'Sorry, there has been an error.', 'impeka-importer' ) . '</strong><br />';
			echo esc_html( $import_data->get_error_message() ) . '</p>';
			$this->footer();
			die();
		}

		$woo_enabled = false;
		$cols = 2;
		if ( class_exists( 'woocommerce' ) ) {
			$woo_enabled = true;
		}

		$page_array = array();
		$post_array = array();
		$portfolio_array = array();
		$product_array = array();
		$area_array = array();
		foreach ($import_data['posts'] as $key => $value) {

			switch ( $value['post_type'] ) {
				case 'page':
				case 'post':
				case 'portfolio':
				case 'product':
				case 'area-item':
					$results = array();
					$results[] = $value['post_id'];

					$regexp = '|ids="([^"]*)"|';
					preg_match_all( $regexp, $value['post_content'], $matches);
					if( isset( $matches[1] ) ) {
						foreach( $matches[1] as $match ) {
							$results[] = $match;
						}
					}
					$regexp = '|image="([^"]*)"|';
					preg_match_all( $regexp, $value['post_content'], $matches);
					if( isset( $matches[1] ) ) {
						foreach( $matches[1] as $match ) {
							$results[] = $match;
						}
					}
					$regexp = '|retina_image="([^"]*)"|';
					preg_match_all( $regexp, $value['post_content'], $matches);
					if( isset( $matches[1] ) ) {
						foreach( $matches[1] as $match ) {
							$results[] = $match;
						}
					}
					$regexp = '|image2="([^"]*)"|';
					preg_match_all( $regexp, $value['post_content'], $matches);
					if( isset( $matches[1] ) ) {
						foreach( $matches[1] as $match ) {
							$results[] = $match;
						}
					}
					$regexp = '|retina_image2="([^"]*)"|';
					preg_match_all( $regexp, $value['post_content'], $matches);
					if( isset( $matches[1] ) ) {
						foreach( $matches[1] as $match ) {
							$results[] = $match;
						}
					}
					$regexp = '|bg_image="([^"]*)"|';
					preg_match_all( $regexp, $value['post_content'], $matches);
					if( isset( $matches[1] ) ) {
						foreach( $matches[1] as $match ) {
							$results[] = $match;
						}
					}
					$regexp = '|contact-form-7 id="([^"]*)"|';
					preg_match_all( $regexp, $value['post_content'], $matches);
					if( isset( $matches[1] ) ) {
						foreach( $matches[1] as $match ) {
							$results[] = $match;
						}
					}

					if (isset($value['postmeta'])) {
						foreach ($value['postmeta'] as $meta_key => $meta_value) {

							if ( '_impeka_grve_feature_section' === $meta_value['key'] ) {
								$section = maybe_unserialize( $meta_value['value'] );
								if ( !empty( $section ) && is_array( $section ) ) {
									$single_image = array_column( $section, 'bg_image_id' );
									$match = implode ( ',' , $single_image );
									if( !empty( $match ) ) {
										$results[] = $match;
									}
								}
								if( isset( $section['slider_items'] ) ) {
									$slider_items = $section['slider_items'];
									if ( !empty( $slider_items ) && is_array( $slider_items ) ) {
										$slider_images = array_column( $slider_items, 'bg_image_id' );
										$match = implode ( ',' , $slider_images );
										if( !empty( $match ) ) {
											$results[] = $match;
										}
									}
								}
							} elseif ( '_impeka_grve_post_slider_items' === $meta_value['key'] || '_impeka_grve_portfolio_slider_items' === $meta_value['key']  ) {
								$items = maybe_unserialize( $meta_value['value'] );
								if ( !empty( $items ) && is_array( $items ) ) {
									$slider_images = array_column( $items, 'id' );
									$match = implode ( ',' , $slider_images );
									if( !empty( $match ) ) {
										$results[] = $match;
									}
								}
							} elseif (
								'_thumbnail_id' === $meta_value['key'] ||
								'_impeka_grve_area_image_id' === $meta_value['key'] ||
								'_impeka_grve_second_featured_image' === $meta_value['key'] ||
								'_product_image_gallery' === $meta_value['key'] ) {
								if( !empty( $meta_value['value'] ) )  {
									$results[] = $meta_value['value'];
								}
							}
						}
					}

					if( 'page' == $value['post_type'] ) {
						$page_array[$value['post_title']] = array(
							'id' => $results,
						);
					} elseif ( 'portfolio' == $value['post_type'] ) {
						$portfolio_array[$value['post_title']] = array(
							'id' => $results,
						);
					} elseif ( 'post' == $value['post_type'] ) {
						$post_array[$value['post_title']] = array(
							'id' => $results,
						);
					} elseif ( 'product' == $value['post_type'] ) {
						$product_array[$value['post_title']] = array(
							'id' => $results,
						);
					} elseif ( 'area-item' == $value['post_type'] ) {
						$area_array[$value['post_title']] = array(
							'id' => $results,
						);
					}

				break;

			}
		}
?>
	<div class="grve-selectors-wrapper">
	<?php if ( !empty( $page_array ) ) { ?>
		<div class="grve-single-selector-wrapper">
			<div class="grve-single-selector-inner">
				<span class="grve-single-label"><?php esc_html_e( 'Pages', 'impeka-importer' ); ?></span>
				<select class="grve-single-selector grve-admin-dummy-option-single-pages" name="grve-admin-option-single-pages[]" multiple>
					<?php
					ksort($page_array);
					foreach ($page_array as $key => $value) {
						echo '<option value="'.esc_attr(implode(',', $value['id'])).'">' . $key . '</option>';
					}
					?>
				</select>
			</div>
		</div>
	<?php } ?>
	<?php if ( !empty( $portfolio_array ) ) { ?>
		<div class="grve-single-selector-wrapper">
			<div class="grve-single-selector-inner">
				<span class="grve-single-label"><?php esc_html_e( 'Portfolio Items', 'impeka-importer' ); ?></span>
				<select class="grve-single-selector grve-admin-dummy-option-single-portfolios" name="grve-admin-option-single-portfolios[]" multiple>
				<?php
				ksort($portfolio_array);
				foreach ($portfolio_array as $key => $value) {
					echo '<option value="'.esc_attr(implode(',', $value['id'])).'">' . $key . '</option>';
				}
				?>
				</select>
			</div>
		</div>
	<?php } ?>
	<?php if ( !empty( $post_array ) ) { ?>
		<div class="grve-single-selector-wrapper">
			<div class="grve-single-selector-inner">
				<span class="grve-single-label"><?php esc_html_e( 'Posts', 'impeka-importer' ); ?></span>
				<select class="grve-single-selector grve-admin-dummy-option-single-posts" name="grve-admin-option-single-posts[]" multiple>
				<?php
				ksort($post_array);
				foreach ($post_array as $key => $value) {
					echo '<option value="'.esc_attr(implode(',', $value['id'])).'">' . $key . '</option>';
				}
				?>
				</select>
			</div>
		</div>
	<?php } ?>
	<?php if ( !empty( $product_array ) && $woo_enabled ) { ?>
		<div class="grve-single-selector-wrapper">
			<div class="grve-single-selector-inner">
				<span class="grve-single-label"><?php esc_html_e( 'Products', 'impeka-importer' ); ?></span>
				<select class="grve-single-selector grve-admin-dummy-option-single-products" name="grve-admin-option-single-products[]" multiple>
				<?php
				ksort($product_array);
				foreach ($product_array as $key => $value) {
					echo '<option value="'.esc_attr(implode(',', $value['id'])).'">' . $key . '</option>';
				}
				?>
				</select>
			</div>
		</div>
	<?php } ?>
	<?php if ( !empty( $area_array ) ) { ?>
		<div class="grve-single-selector-wrapper">
			<div class="grve-single-selector-inner">
				<span class="grve-single-label"><?php esc_html_e( 'Area Items', 'impeka-importer' ); ?></span>
				<select class="grve-single-selector grve-admin-dummy-option-single-areas" name="grve-admin-option-single-areas[]" multiple>
				<?php
				ksort($area_array);
				foreach ($area_array as $key => $value) {
					echo '<option value="'.esc_attr(implode(',', $value['id'])).'">' . $key . '</option>';
				}
				?>
				</select>
			</div>
		</div>
	<?php } ?>
	</div>
<?php
	}

	/**
	 * Print Featured Plugins
 	 */
	function grve_print_featured_plugins( $featured_plugins = array() ) {
		$grve_plugin_data = $this->grve_get_plugin_data();
		if( !empty( $featured_plugins ) ) {
			echo '<div class="grve-demo-plugins">';
			foreach($featured_plugins as $plugin_slug ) {
				if( isset( $grve_plugin_data[$plugin_slug] ) ) {
					$image_src = plugins_url( '/assets/images/plugins/' . $plugin_slug . '.png', __FILE__ );
				?>
					<div class="grve-demo-plugin">
						<img src="<?php echo esc_url( $image_src ); ?>" alt="<?php echo esc_attr( $grve_plugin_data[$plugin_slug]['name'] ); ?>"/>
						<?php echo esc_html( $grve_plugin_data[$plugin_slug]['name'] ); ?>
					</div>
				<?php
				}
			}
			echo '</div>';
		}
	}

	/**
	 * Get Inactive Plugins
 	 */
	function grve_get_inactive_plugins( $required_plugins = array() ) {
		$inactive_plugins = array();
		$grve_plugin_data = $this->grve_get_plugin_data();
		if( !empty( $required_plugins ) ) {
			foreach($required_plugins as $plugin_slug ) {
				if( isset( $grve_plugin_data[$plugin_slug] ) ) {
					if ( ! $grve_plugin_data[$plugin_slug]['active'] ) {
						$inactive_plugins[] = $grve_plugin_data[$plugin_slug]['name'];
					}
				}
			}
		}
		return $inactive_plugins;
	}

	/**
	 * Get Demo data by ID
 	 */
	function grve_get_demo_data_by_id( $dummy_id ) {

		$grve_dummy_data_list = $this->grve_get_demo_data();
		$dummy_data = array(
			'id' => 'undefined',
			'homepage' => 'Home',
			'impeka_header_nav'  => 'main-menu',
		);

		foreach ( $grve_dummy_data_list as $grve_dummy_data ) {
			if ( $dummy_id == $grve_dummy_data['id'] ) {
				$dummy_data = $grve_dummy_data;
				break;
			}
		}

		return $dummy_data;
	}

	/**
	 * Remove Image Sizes to improve speed
 	 */
	function impeka_import_remove_image_sizes() {
		remove_image_size( 'impeka-grve-fullscreen');
	}

	/**
	 * Imports Dummy Attachements
 	 */
	function impeka_import_attachments() {

		$importer_info = '';
		$eut_index  = $eut_total = 0;

		if ( isset( $_POST['grve_import_data'] ) ) {
			ob_start();
			$dummy_id = $_POST['grve_import_data'];
			$grve_importer_error = false;
			$grve_changed = $grve_finished = false;
			check_ajax_referer( $dummy_id, 'nonce', true );
			$grve_theme_active = $this->is_theme_active();
			if ( $grve_theme_active && !version_compare( phpversion(), IMPEKA_IMPORT_MIN_PHP_VERSION, '<' ) ) {
				//Import Partial Attachments
				if ( isset( $_POST['grve_import_attachments'] ) && 'true' == $_POST['grve_import_attachments'] ) {

					if ( isset( $_POST['grve_import_index'] ) ) {
						$grve_index = intval ( $_POST['grve_import_index'] );
					}
					if ( isset( $_POST['grve_import_demo_images'] ) && 'false' == $_POST['grve_import_demo_images'] ) {
						$grve_index_step = 50;
					} else {
						$grve_index_step = 10;
					}

					$import_file = plugin_dir_path(__FILE__) . 'import/data/' . $dummy_id  .  '/dummy.xml';
					if ( ! file_exists( $import_file ) ) {
						$import_file = plugin_dir_path(__FILE__) . 'import/data/' . $dummy_id  .  '/dummy.xml.gz';
					}
					$import_data = $this->parse( $import_file );
					$this->posts = $import_data['posts'];
					$attachments = 0;
					$attachment_ids = array();

					$index_step = $grve_index + $grve_index_step;

					if ( isset( $_POST['grve_import_singular'] ) && 'true' == $_POST['grve_import_singular'] ) {
						if (
							( isset( $_POST['grve_import_single_pages'] ) && !empty( $_POST['grve_import_single_pages'] ) ) ||
							( isset( $_POST['grve_import_single_posts'] ) && !empty( $_POST['grve_import_single_posts'] ) ) ||
							( isset( $_POST['grve_import_single_portfolios'] ) && !empty( $_POST['grve_import_single_portfolios'] ) ) ||
							( isset( $_POST['grve_import_single_products'] ) && !empty( $_POST['grve_import_single_products'] ) ) ||
							( isset( $_POST['grve_import_single_areas'] ) && !empty( $_POST['grve_import_single_areas'] ) )
						) {

							$ids = array();
							$ids1 = $_POST['grve_import_single_pages'];
							$ids2 = $_POST['grve_import_single_posts'];
							$ids3 = $_POST['grve_import_single_portfolios'];
							$ids4 = $_POST['grve_import_single_products'];
							$ids5 = $_POST['grve_import_single_areas'];

							if( !empty( $ids1 ) ) {
								$ids = array_merge($ids, $ids1);
							}
							if( !empty( $ids2 ) ) {
								$ids = array_merge($ids, $ids2);
							}
							if( !empty( $ids3 ) ) {
								$ids = array_merge($ids, $ids3);
							}
							if( !empty( $ids4 ) ) {
								$ids = array_merge($ids, $ids4);
							}
							if( !empty( $ids5 ) ) {
								$ids = array_merge($ids, $ids5);
							}

							if ( !empty( $ids ) ) {
								$ids_array = array();
								$ids_string = implode ( ',' , $ids  );
								$ids_array = explode( ',' , $ids_string);
								$ids_array = array_unique( $ids_array );
							} else {
								$ids_array = array();
							}

							foreach ($import_data['posts'] as $key => $value) {
								if ( 'attachment' == $value['post_type'] ) {
									if (in_array( $value['post_id'], $ids_array)) {
										if ( $attachments >= $grve_index && $attachments < $index_step ) {
											$attachment_ids[] = $value['post_id'];
										}
										$attachments++;
									}
								}
							}
						}
					} else {
						foreach ($import_data['posts'] as $key => $value) {
							if ( 'attachment' == $value['post_type'] ) {

								if ( $attachments >= $grve_index && $attachments < $index_step ) {
									$attachment_ids[] = $value['post_id'];
								}
								$attachments++;
							}
						}
					}
					$grve_index = $index_step;
					$grve_total = $attachments;

					if ( $grve_index >= $grve_total ) {
						$grve_finished = true;
						$grve_index = $grve_total;
					}

					set_time_limit(0);
					$this->fetch_attachments = true;
					if ( isset( $_POST['grve_import_demo_images'] ) && 'true' == $_POST['grve_import_demo_images'] ) {
						$this->fetch_live_images = true;
						$this->impeka_import_remove_image_sizes();
					}
					if ( !empty( $attachment_ids ) ) {
						$this->process_posts( $attachment_ids );
					}
					if ( isset( $_POST['grve_import_demo_images'] ) && 'false' == $_POST['grve_import_demo_images'] ) {
						$importer_info.= esc_html__( "Dummy Attachments Imported", 'impeka-importer' ) . " " . $grve_index . "/" . $grve_total . "<br>";
					} else {
						$importer_info.= esc_html__( "Attachments Imported", 'impeka-importer' ) . " " . $grve_index . "/" . $grve_total . "<br>";
					}
					$grve_changed = true;
				}

				if ( !$grve_importer_error ) {
					if( $grve_finished ) {
						$importer_info.=  '<br/>';
						$importer_info.=  '<i class="dashicons dashicons-yes"></i> ' .  esc_html__( "Attachments:  Imported!", 'impeka-importer' ) . '<br/>';
					}
				} else {
					$importer_info.=  '<br/>';
					$importer_info.=  '<i class="dashicons dashicons-no"></i> <b>' .  esc_html__( "Attachment Import finished with errors!", 'impeka-importer' ) . '</b><br/>';
				}
			} else {
					$grve_importer_error = true;
					if ( version_compare( phpversion(), IMPEKA_IMPORT_MIN_PHP_VERSION, '<' ) ) {
						$importer_info.=  '<i class="dashicons dashicons-no"></i> <b>' .  esc_html__( "The minimum PHP version required for the Dummy Importer is:", 'impeka-importer' ) . ' ' . IMPEKA_IMPORT_MIN_PHP_VERSION . '</b>';
						$importer_info.=  '<br/>';
					}
					if ( !$grve_theme_active ) {
						$importer_info.=  '<i class="dashicons dashicons-no"></i> <b>' .  esc_html__( "Impeka Theme is not activated! Impeka Theme needs to be installed and activated!", 'impeka-importer' ) . '</b>';
						$importer_info.=  '<br/><br/>';
						$importer_info.=  '<a href="' . esc_url( admin_url( 'themes.php' ) ) . '" target="_blank">' .  esc_html__( "Themes", 'impeka-importer' ) . '</a>';
					}
			}

			$importer_output = "";
			$importer_debug_output = ob_get_clean();

			if ( defined('IMPEKA_IMPORT_DEBUG') && IMPEKA_IMPORT_DEBUG ) {
				$importer_output = $importer_debug_output;
			}

			$response = array(
				'errors' => $grve_importer_error,
				'changed' => $grve_changed,
				'info' => $importer_info,
				'output' => $importer_output,
				'index' => $grve_index,
				'total' => $grve_total,
				'finished' => $grve_finished,
			);
			wp_send_json( $response );

		}
		if ( isset( $_POST['grve_import_data'] ) ) { die(); }
	}

	/**
	 * Imports Demo Data
 	 */
	function impeka_import_demo_data() {

		$importer_info = '';

		if ( isset( $_POST['grve_import_data'] ) ) {

			ob_start();

			$dummy_id = $_POST['grve_import_data'];
			$grve_importer_error = false;
			$grve_changed = false;
			check_ajax_referer( $dummy_id, 'nonce', true );
			echo '<br />';

			$grve_theme_active = $this->is_theme_active();

			if ( $grve_theme_active && !version_compare( phpversion(), IMPEKA_IMPORT_MIN_PHP_VERSION, '<' ) ) {

				//Import Singular Data
				if ( isset( $_POST['grve_import_singular'] ) && 'true' == $_POST['grve_import_singular'] ) {
					$import_file = plugin_dir_path(__FILE__) . 'import/data/' . $dummy_id  .  '/dummy.xml';

					if ( ! file_exists( $import_file ) ) {
						$import_file = plugin_dir_path(__FILE__) . 'import/data/' . $dummy_id  .  '/dummy.xml.gz';
					}

					if ( ! file_exists( $import_file ) || empty( $import_file ) ) {
						$grve_importer_error = true;
						$importer_info.=  '<i class="dashicons dashicons-no"></i> ' . esc_html__( "Single Content: File empty ot not existing!", 'impeka-importer' );
						$importer_info.=  '<br />';
					} else {
						if (
							( isset( $_POST['grve_import_single_pages'] ) && !empty( $_POST['grve_import_single_pages'] ) ) ||
							( isset( $_POST['grve_import_single_posts'] ) && !empty( $_POST['grve_import_single_posts'] ) ) ||
							( isset( $_POST['grve_import_single_portfolios'] ) && !empty( $_POST['grve_import_single_portfolios'] ) ) ||
							( isset( $_POST['grve_import_single_products'] ) && !empty( $_POST['grve_import_single_products'] ) ) ||
							( isset( $_POST['grve_import_single_areas'] ) && !empty( $_POST['grve_import_single_areas'] ) )
						) {
							$grve_changed = true;
							set_time_limit(0);
							$this->fetch_attachments = true;
							if ( isset( $_POST['grve_import_demo_images'] ) && 'true' == $_POST['grve_import_demo_images'] ) {
								$this->fetch_live_images = true;
								$this->impeka_import_remove_image_sizes();
							}

							$ids = array();
							$ids1 = $_POST['grve_import_single_pages'];
							$ids2 = $_POST['grve_import_single_posts'];
							$ids3 = $_POST['grve_import_single_portfolios'];
							$ids4 = $_POST['grve_import_single_products'];
							$ids5 = $_POST['grve_import_single_areas'];

							if( !empty( $ids1 ) ) {
								$ids = array_merge($ids, $ids1);
							}
							if( !empty( $ids2 ) ) {
								$ids = array_merge($ids, $ids2);
							}
							if( !empty( $ids3 ) ) {
								$ids = array_merge($ids, $ids3);
							}
							if( !empty( $ids4 ) ) {
								$ids = array_merge($ids, $ids4);
							}
							if( !empty( $ids5 ) ) {
								$ids = array_merge($ids, $ids5);
							}

							$import_output = $this->import( $import_file, $ids );
							if ( is_wp_error( $import_output ) ) {
								$grve_importer_error = true;
								$importer_info.=  '<i class="dashicons dashicons-no"></i> ' . esc_html__( "Single Content: Error During Import!", 'impeka-importer' );
								$importer_info.=  '<br />';
							} else {
								$importer_info.=  '<i class="dashicons dashicons-yes"></i> ' . esc_html__( "Single Content: imported!", 'impeka-importer' );
								$importer_info.=  '<br />';
							}

						}
					}
				}

				//Import Dummy Data
				if ( isset( $_POST['grve_import_content'] ) && 'true' == $_POST['grve_import_content'] ) {
					$grve_changed = true;
					$import_file = plugin_dir_path(__FILE__) . 'import/data/' . $dummy_id  .  '/dummy.xml';

					if ( ! file_exists( $import_file ) ) {
						$import_file = plugin_dir_path(__FILE__) . 'import/data/' . $dummy_id  .  '/dummy.xml.gz';
					}

					if ( ! file_exists( $import_file ) || empty( $import_file ) ) {
						$grve_importer_error = true;
						$importer_info.=  '<i class="dashicons dashicons-no"></i> ' . esc_html__( "Dummy Content: File empty ot not existing!", 'impeka-importer' );
						$importer_info.=  '<br />';
					} else {
						set_time_limit(0);
						$this->fetch_attachments = true;
						if ( isset( $_POST['grve_import_demo_images'] ) && 'true' == $_POST['grve_import_demo_images'] ) {
							$this->fetch_live_images = true;
							$this->impeka_import_remove_image_sizes();
						}

						$import_output = $this->import( $import_file );
						if ( is_wp_error( $import_output ) ) {
							$grve_importer_error = true;
							$importer_info.=  '<i class="dashicons dashicons-no"></i> ' . esc_html__( "Dummy Content: Error During Import!", 'impeka-importer' );
							$importer_info.=  '<br />';
						} else {

							$grve_dummy_data = $this->grve_get_demo_data_by_id( $dummy_id );

							//Set Menu Locations
							$grve_menus  = wp_get_nav_menus();
							$locations = get_theme_mod( 'nav_menu_locations' );
							if( ! empty( $grve_menus ) ) {
								foreach ( $grve_menus as $grve_menu ) {
									if( isset( $grve_dummy_data['impeka_header_nav'] ) && $grve_dummy_data['impeka_header_nav'] == $grve_menu->slug ) {
										$locations['impeka_header_nav'] = $grve_menu->term_id;
									}
									if( isset( $grve_dummy_data['impeka_header_second_nav'] ) && $grve_dummy_data['impeka_header_second_nav'] == $grve_menu->slug ) {
										$locations['impeka_header_second_nav'] = $grve_menu->term_id;
									}
									if( isset( $grve_dummy_data['impeka_responsive_nav'] ) && $grve_dummy_data['impeka_responsive_nav'] == $grve_menu->slug ) {
										$locations['impeka_responsive_nav'] = $grve_menu->term_id;
									}
									if( isset( $grve_dummy_data['impeka_top_left_nav'] ) && $grve_dummy_data['impeka_top_left_nav'] == $grve_menu->slug ) {
										$locations['impeka_top_left_nav'] = $grve_menu->term_id;
									}
									if( isset( $grve_dummy_data['impeka_top_right_nav'] ) && $grve_dummy_data['impeka_top_right_nav'] == $grve_menu->slug ) {
										$locations['impeka_top_right_nav'] = $grve_menu->term_id;
									}
									if( isset( $grve_dummy_data['impeka_footer_nav'] ) && $grve_dummy_data['impeka_footer_nav'] == $grve_menu->slug ) {
										$locations['impeka_footer_nav'] = $grve_menu->term_id;
									}
								}
								set_theme_mod( 'nav_menu_locations', $locations );
							}

							//Set Home page
							$homepage = '';
							if( isset( $grve_dummy_data['homepage'] ) && !empty( $grve_dummy_data['homepage'] ) ) {
								$homepage = get_page_by_title( $grve_dummy_data['homepage'] );
							}
							if ( $homepage ) {
								update_option( 'page_on_front', $homepage->ID );
								update_option( 'show_on_front', 'page' );
							}

							//Set Posts page
							$postspage = '';
							if( isset( $grve_dummy_data['postspage'] ) && !empty( $grve_dummy_data['postspage'] ) ) {
								$postspage = get_page_by_title( $grve_dummy_data['postspage'] );
							}
							if ( $postspage ) {
								update_option( 'page_for_posts', $postspage->ID );
							}

							//Import Revolution Slider
							if ( class_exists('RevSlider') ) {

								$import_file = plugin_dir_path(__FILE__) . 'import/data/' . $dummy_id  .  '/revslider1.zip';
								if ( ! file_exists( $import_file ) || empty( $import_file ) ) {
									//No revolution slider available for this demo
								} else {
									$revslider = new RevSlider();
									$revslider->importSliderFromPost( false, false, $import_file );
								}

								$import_file = plugin_dir_path(__FILE__) . 'import/data/' . $dummy_id  .  '/revslider2.zip';
								if ( ! file_exists( $import_file ) || empty( $import_file ) ) {
									//No revolution slider available for this demo
								} else {
									$revslider = new RevSlider();
									$revslider->importSliderFromPost( false, false, $import_file );
								}

								$import_file = plugin_dir_path(__FILE__) . 'import/data/' . $dummy_id  .  '/revslider3.zip';
								if ( ! file_exists( $import_file ) || empty( $import_file ) ) {
									//No revolution slider available for this demo
								} else {
									$revslider = new RevSlider();
									$revslider->importSliderFromPost( false, false, $import_file );
								}
							}

							$importer_info.=  '<i class="dashicons dashicons-yes"></i> ' . esc_html__( "Dummy Content: imported!", 'impeka-importer' );
							$importer_info.=  '<br />';
						}
					}
				}

				//Import Theme Options
				if ( isset( $_POST['grve_import_options'] ) && 'true' == $_POST['grve_import_options'] ) {
					$grve_changed = true;
					$import_file = plugin_dir_path(__FILE__) . 'import/data/' . $dummy_id  .  '/grve-options.json';
					if ( ! file_exists( $import_file ) || empty( $import_file ) ) {
						$grve_importer_error = true;
						$importer_info.=  '<i class="dashicons dashicons-no"></i> ' . esc_html__( "Theme Options: file empty ot not existing!", 'impeka-importer' );
						$importer_info.=  '<br/>';
					} else {
						if ( $this->grve_import_options( $import_file, $dummy_id ) ) {
							$importer_info.=  '<i class="dashicons dashicons-yes"></i> ' . esc_html__( "Theme Options: imported!", 'impeka-importer' );
							$importer_info.=  '<br/>';
						}
					}
				}

				//Import Widgets
				if ( isset( $_POST['grve_import_widgets'] ) && 'true' == $_POST['grve_import_widgets'] ) {
					$grve_changed = true;
					$import_file = plugin_dir_path(__FILE__) . 'import/data/' . $dummy_id  .  '/widget_data.json';
					if ( ! file_exists( $import_file ) || empty( $import_file ) ) {
						//$importer_info.=  '<i class="dashicons dashicons-info"></i> ' . esc_html__( "Widgets: no widgets available for this Demo!", 'impeka-importer' );
						//$importer_info.=  '<br/>';
					} else {
						if ( $this->grve_import_demo_widgets( $import_file ) ) {
							$importer_info.=  '<i class="dashicons dashicons-yes"></i> ' . esc_html__( "Widgets: imported!", 'impeka-importer' );
							$importer_info.=  '<br/>';
						}
					}
				}

				//Import Attachments
				if ( isset( $_POST['grve_import_demo_images'] ) && 'true' == $_POST['grve_import_demo_images'] ) {
					$grve_changed = true;
				}

				if ( !$grve_importer_error ) {
					if( $grve_changed ) {
						$importer_info.=  '<br/>';
						$importer_info.=  '<i class="dashicons dashicons-yes"></i> <b>' .  esc_html__( "Import finished!", 'impeka-importer' ) . '</b>';
						$importer_info.=  '<br/><br/>';
						if ( defined('IMPEKA_IMPORT_DEBUG') && IMPEKA_IMPORT_DEBUG ) {
							$importer_info.=  '<a href="' . esc_url( home_url() ) . '" target="_blank">' .  esc_html__( "Visit Site", 'impeka-importer' ) . '</a>';
						} else {
							$importer_info.=  esc_html__( "Redirecting to finish page...", 'impeka-importer' );
						}
					} else {
						$importer_info.=  '<i class="dashicons dashicons-info"></i> <b>' .  esc_html__( "No options selected, please select some options and press the import button!", 'impeka-importer' ) . '</b>';
						$importer_info.=  '<br/>';
					}
				} else {
					$importer_info.=  '<br/>';
					$importer_info.=  '<i class="dashicons dashicons-no"></i> <b>' .  esc_html__( "Import finished with errors!", 'impeka-importer' ) . '</b>';
					$importer_info.=  ' <a href="' . esc_url( home_url( '/' ) ) . '">' .  esc_html__( "Visit Site", 'impeka-importer' ) . '</a>';
					$importer_info.=  '<br/>';
				}
			} else {

					if ( version_compare( phpversion(), IMPEKA_IMPORT_MIN_PHP_VERSION, '<' ) ) {
						$importer_info.=  '<i class="dashicons dashicons-no"></i> <b>' .  esc_html__( "The minimum PHP version required for the Dummy Importer is:", 'impeka-importer' ) . ' ' . IMPEKA_IMPORT_MIN_PHP_VERSION . '</b>';
						$importer_info.=  '<br/>';
					}

					if ( !$grve_theme_active ) {
						$importer_info.=  '<i class="dashicons dashicons-no"></i> <b>' .  esc_html__( "Impeka Theme is not activated! Impeka Theme needs to be installed and activated!", 'impeka-importer' ) . '</b>';
						$importer_info.=  '<br/>';
					}

			}

			$importer_output = "";
			$importer_debug_output = ob_get_clean();

			if ( defined('IMPEKA_IMPORT_DEBUG') && IMPEKA_IMPORT_DEBUG ) {
				$importer_output = $importer_debug_output;
			}

			$response = array(
				'errors' => $grve_importer_error,
				'changed' => $grve_changed,
				'info' => $importer_info,
				'output' => $importer_output,
			);
			wp_send_json( $response );

		}
		if ( isset( $_POST['grve_import_data'] ) ) { die(); }
	}

	/**
	 * Additional function to get a new widget name
	 * Used from grve_import_demo_widgets
 	 */
	function grve_get_new_widget_name( $widget_name, $widget_index ) {
		$current_sidebars = get_option( 'sidebars_widgets' );
		$all_widget_array = array( );
		foreach ( $current_sidebars as $sidebar => $widgets ) {
			if ( ! empty( $widgets ) && is_array( $widgets ) && 'wp_inactive_widgets' != $sidebar ) {
				foreach ( $widgets as $widget ) {
					$all_widget_array[] = $widget;
				}
			}
		}
		while ( in_array( $widget_name . '-' . $widget_index, $all_widget_array ) ) {
			$widget_index++;
		}
		$new_widget_name = $widget_name . '-' . $widget_index;
		return $new_widget_name;
	}

	/**
	 * Imports widgets from file
 	 */
	function grve_import_demo_widgets( $import_file ) {

		if ( file_exists( $import_file ) ){

			$import_array = file_get_contents( $import_file );
			$import_array = json_decode( $import_array, true );

			$sidebars_data = $import_array[0];
			$widget_data = $import_array[1];
			$new_widgets = array( );

			//Get Existing Custom sidebars
			$grve_custom_sidebars = get_option( '_impeka_grve_custom_sidebars' );
			if ( empty( $grve_custom_sidebars ) ) {
				$grve_custom_sidebars = array();
			}

			$custom_sidebars_ids = array();
			if ( ! empty( $grve_custom_sidebars ) ) {
				foreach ( $grve_custom_sidebars as $grve_custom_sidebar ) {
					array_push( $custom_sidebars_ids, $grve_custom_sidebar['id'] );
				}
			}

			$current_sidebars = wp_get_sidebars_widgets();

			$current_sidebars['grve-default-sidebar'] = array();
			$current_sidebars['grve-single-portfolio-sidebar'] = array();
			$current_sidebars['grve-footer-1-sidebar'] = array();
			$current_sidebars['grve-footer-2-sidebar'] = array();
			$current_sidebars['grve-footer-3-sidebar'] = array();
			$current_sidebars['grve-footer-4-sidebar'] = array();

			//Check if includes custom sidebars
			$sidebar_index = 0;
			$new_sidebars = false;

			foreach ( $sidebars_data as $import_sidebar => $import_widgets ) :

				if( strpos( $import_sidebar, "impeka_grve_sidebar_" ) !== false ) {
					if ( !in_array( $import_sidebar, $custom_sidebars_ids ) ) {
						$sidebar_index++;
						$this_sidebar = array ( 'id' => $import_sidebar , 'name' => "Demo Sidebar " . $sidebar_index );
						array_push( $grve_custom_sidebars, $this_sidebar );
						if( ! isset( $current_sidebars[ $import_sidebar ] ) ) {
							$current_sidebars[ $import_sidebar ] = array();
						}
						$new_sidebars = true;
					}
				}

			endforeach;

			//Update and Register Custom Sidebars if needed
			if ( ! empty( $grve_custom_sidebars ) && $new_sidebars ) {
				update_option( '_impeka_grve_custom_sidebars', $grve_custom_sidebars );
				wp_set_sidebars_widgets( $current_sidebars );
			}

			//Get Current Sidebars and Widgets
			$current_sidebars = wp_get_sidebars_widgets();

			//Import Widget Data
			foreach ( $sidebars_data as $import_sidebar => $import_widgets ) :

				foreach ( $import_widgets as $import_widget ) :
					//if the sidebar exists
					if ( isset( $current_sidebars[$import_sidebar] ) ) :
						$title = trim( substr( $import_widget, 0, strrpos( $import_widget, '-' ) ) );
						$index = trim( substr( $import_widget, strrpos( $import_widget, '-' ) + 1 ) );
						$current_widget_data = get_option( 'widget_' . $title );
						$new_widget_name = $this->grve_get_new_widget_name( $title, $index );
						$new_index = trim( substr( $new_widget_name, strrpos( $new_widget_name, '-' ) + 1 ) );

						if ( ! empty( $new_widgets[ $title ] ) && is_array( $new_widgets[ $title ] ) ) {
							while ( array_key_exists( $new_index, $new_widgets[ $title ] ) ) {
								$new_index++;
							}
						}
						$current_sidebars[ $import_sidebar ][] = $title . '-' . $new_index;
						if ( array_key_exists( $title, $new_widgets ) ) {
							$new_widgets[$title][ $new_index ] = $widget_data[ $title ][ $index ];
							$multiwidget = $new_widgets[ $title ]['_multiwidget'];
							unset( $new_widgets[ $title ]['_multiwidget'] );
							$new_widgets[ $title ]['_multiwidget'] = $multiwidget;
						} else {
							$current_widget_data[ $new_index ] = $widget_data[ $title ][ $index ];
							$current_multiwidget = '';
							if ( isset($current_widget_data['_multiwidget'] ) ) {
								$current_multiwidget = $current_widget_data['_multiwidget'];
							}
							$new_multiwidget = $widget_data[ $title ]['_multiwidget'];
							$multiwidget = ( $current_multiwidget != $new_multiwidget ) ? $current_multiwidget : 1;
							unset( $current_widget_data['_multiwidget'] );
							$current_widget_data['_multiwidget'] = $multiwidget;
							$new_widgets[ $title ] = $current_widget_data;
						}

					endif;
				endforeach;
			endforeach;

			if ( isset( $new_widgets ) && isset( $current_sidebars ) ) {
				wp_set_sidebars_widgets( $current_sidebars );

				foreach ( $new_widgets as $title => $content )
					update_option( 'widget_' . $title, $content );

				return true;
			}

			return false;

		}
		else{
			return false; //Widget File not found
		}
	}

	/**
	 * Get Plugin info
 	 */
	function grve_get_plugin_data() {
		$plugin_data = array(
			'impeka-extension' => array(
				'name' => esc_html__( 'Impeka Core Extension', 'impeka-importer' ),
				'active' => class_exists( 'Impeka_Extension_Plugin' ),
			),
			'gutenberg' => array(
				'name' => esc_html__( 'Gutenberg', 'impeka-importer' ),
				'active' => true,
			),
			'impeka-gutenberg-extension' => array(
				'name' => esc_html__( 'Impeka Gutenberg Extension', 'impeka-importer' ),
				'active' => class_exists( 'Impeka_Gutenberg_Extension_Plugin' ),

			),
			'js_composer' => array(
				'name' => esc_html__( 'WPBakery Page Builder', 'impeka-importer' ),
				'active' => class_exists( 'WPBakeryShortCode' ),
			),
			'impeka-wpb-extension' => array(
				'name'=> esc_html__( 'Impeka WPBakery Extension', 'impeka-importer' ),
				'active' => class_exists( 'Impeka_WPBakery_Extension_Plugin' ),
			),
			'elementor' => array(
				'name' => esc_html__( 'Elementor', 'impeka-importer' ),
				'active' => class_exists( 'Elementor\Widget_Base' ),
			),
			'impeka-elementor-extension' => array(
				'name' => esc_html__( 'Impeka Elementor Extension', 'impeka-importer' ),
				'active' => class_exists( 'Impeka_Elementor_Extension_Plugin' ),
			),
			'woocommerce' => array(
				'name' => esc_html__( 'WooCommerce', 'impeka-importer' ),
				'active' => class_exists( 'woocommerce' ),
			),
			'the-events-calendar' => array(
				'name' => esc_html__( 'The Events Calendar', 'impeka-importer' ),
				'active' => class_exists( 'Tribe__Events__Main' ),
			),
		);
		return $plugin_data;
	}

	/**
	 * Check if theme registered
 	 */
	function is_theme_registered() {
		if ( function_exists( 'impeka_grve_check_theme_status') ) {
			return impeka_grve_check_theme_status( 'import' );
		}
		return false;
	}

	/**
	 * Check if theme active
 	 */
	function is_theme_active() {
		if ( function_exists( 'impeka_grve_info') ) {
			return true;
		}
		return false;
	}

	/**
	 * Get Demo data values
 	 */
	function grve_get_demo_data() {
		$grve_dummy_data_list = array(
			array(
				'id'   => 'impeka',
				'name' => esc_html__( 'Impeka Main', 'impeka-importer' ),
				'dir'  => 'impeka',
				'preview'  => 'impeka',
				'description'  => esc_html__( 'Several sample pages and layouts to get inspired. Elements are also included here to get a simple understanding of how Impeka works.', 'impeka-importer' ),
				'homepage' => 'Homepage I',
				'impeka_header_nav'  => 'main-menu',
				'impeka_header_second_nav'  => 'second-header-menu',
				'featured_plugins' => array( 'js_composer' ),
				'required_plugins' => array( 'impeka-extension', 'js_composer', 'impeka-wpb-extension' ),
				'single_selector' => true,
			),
			array(
				'id'   => 'impeka-shop',
				'name' => esc_html__( 'Impeka Shop', 'impeka-importer' ),
				'dir'  => 'impeka-shop',
				'preview'  => 'impeka/impeka-shop',
				'description'  => esc_html__( 'Shop case study with WooCommerce products and Impeka. Showcase your products and start selling in a few minutes.', 'impeka-importer' ),
				'homepage' => 'Homepage',
				'impeka_header_nav'  => 'main-menu',
				'impeka_header_second_nav'  => 'second-menu',
				'impeka_responsive_nav'  => 'main-menu',
				'featured_plugins' => array( 'js_composer', 'woocommerce' ),
				'required_plugins' => array( 'impeka-extension', 'js_composer', 'impeka-wpb-extension', 'woocommerce' ),
				'single_selector' => true,
			),
			array(
				'id'   => 'impeka-intro',
				'name' => esc_html__( 'Impeka Intro', 'impeka-importer' ),
				'dir'  => 'impeka-intro',
				'preview'  => 'impeka/intro',
				'description'  => esc_html__( 'The impressive landing page of Impeka to get inspired. A handy section navigation and several background effects.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'impeka_header_second_nav'  => 'second-header-menu',
				'featured_plugins' => array( 'js_composer' ),
				'required_plugins' => array( 'impeka-extension', 'js_composer', 'impeka-wpb-extension' ),
			),
			array(
				'id'   => 'impeka-intro-elementor',
				'name' => esc_html__( 'Impeka Intro Elementor', 'impeka-importer' ),
				'dir'  => 'impeka-intro-elementor',
				'preview'  => 'impeka/intro-elementor',
				'description'  => esc_html__( 'The impressive landing page of Impeka to get inspired. A handy section navigation and several background effects.', 'impeka-importer' ),
				'homepage' => 'Intro Page',
				'impeka_header_second_nav'  => 'second-menu',
				'featured_plugins' => array( 'elementor' ),
				'required_plugins' => array( 'impeka-extension', 'elementor', 'impeka-elementor-extension' ),
			),
			array(
				'id'   => 'impeka-shop-v2',
				'name' => esc_html__( 'Impeka Shop V2', 'impeka-importer' ),
				'dir'  => 'impeka-shop-v2',
				'preview'  => 'impeka/impeka-shop-v2',
				'description'  => esc_html__( 'Another shop case study with WooCommerce products, Elementor builder. Showcase your products and start selling in a few minutes.', 'impeka-importer' ),
				'homepage' => 'Home',
				'postspage' => 'Journal',
				'impeka_header_nav'  => 'main-menu',
				'impeka_footer_nav' => 'footer-menu',
				'featured_plugins' => array( 'elementor', 'woocommerce' ),
				'required_plugins' => array( 'impeka-extension', 'elementor', 'impeka-elementor-extension', 'woocommerce' ),
			),
			array(
				'id'   => 'impeka-corporate',
				'name' => esc_html__( 'Impeka Corporate v1', 'impeka-importer' ),
				'dir'  => 'impeka-corporate',
				'preview'  => 'impeka/impeka-corporate-v1',
				'description'  => esc_html__( 'Modern, creative and clean case study for design agencies, startups and freelancers. Created with Impeka Enhanced Page Builder.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'js_composer' ),
				'required_plugins' => array( 'impeka-extension', 'js_composer', 'impeka-wpb-extension' ),
				'single_selector' => true,
			),
			array(
				'id'   => 'impeka-coffee-shop',
				'name' => esc_html__( 'Impeka Coffee Shop', 'impeka-importer' ),
				'dir'  => 'impeka-coffee-shop',
				'preview'  => 'impeka/impeka-coffee-shop',
				'description'  => esc_html__( 'An impressive one page created with Elementor page builder. The Skin on Scroll template dominates on this Impeka case.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'elementor' ),
				'required_plugins' => array( 'impeka-extension', 'elementor', 'impeka-elementor-extension' ),
			),
			array(
				'id'   => 'impeka-photographer',
				'name' => esc_html__( 'Impeka Photographer Demo', 'impeka-importer' ),
				'dir'  => 'impeka-photographer',
				'preview'  => 'impeka/impeka-photographer',
				'description'  => esc_html__( 'Dark style for creative photographers and freelancers. Created with Gutenberg Editor, the default WP editor.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'gutenberg' ),
				'required_plugins' => array( 'impeka-extension', 'impeka-gutenberg-extension' ),
			),
			array(
				'id'   => 'impeka-architecture',
				'name' => esc_html__( 'Impeka Architecture', 'impeka-importer' ),
				'dir'  => 'impeka-architecture',
				'preview'  => 'impeka/impeka-architecture',
				'description'  => esc_html__( 'Handcrafted for Architects, Engineering and Construction companies. Created with Impeka Enhanced Page Builder.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'js_composer' ),
				'required_plugins' => array( 'impeka-extension', 'js_composer', 'impeka-wpb-extension' ),
				'single_selector' => true,
			),
			array(
				'id'   => 'impeka-restaurant',
				'name' => esc_html__( 'Impeka Restaurant', 'impeka-importer' ),
				'dir'  => 'impeka-restaurant',
				'preview'  => 'impeka/impeka-restaurant',
				'description'  => esc_html__( 'Real Case study for restaurants, bistros, coffee shops and similar projects. Created with Impeka Enhanced Page Builder.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'js_composer' ),
				'required_plugins' => array( 'impeka-extension', 'js_composer', 'impeka-wpb-extension' ),
			),
			array(
				'id'   => 'impeka-clinic',
				'name' => esc_html__( 'Impeka Medical', 'impeka-importer' ),
				'dir'  => 'impeka-clinic',
				'preview'  => 'impeka/impeka-clinic',
				'description'  => esc_html__( 'Medical case study with Modern Tribe Events calendar. Simple and clean design, created with Elementor page builder.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'elementor', 'the-events-calendar' ),
				'required_plugins' => array( 'impeka-extension', 'elementor', 'impeka-elementor-extension', 'the-events-calendar' ),
			),
			array(
				'id'   => 'impeka-blog',
				'name' => esc_html__( 'Impeka Blog', 'impeka-importer' ),
				'dir'  => 'impeka-blog',
				'preview'  => 'impeka/impeka-blog',
				'description'  => esc_html__( 'If you are a Blogger and publish regularly, you will love Impeka Blog. Created with Gutenberg Editor, the default WP editor.', 'impeka-importer' ),
				'homepage' => 'Home',
				'postspage' => 'Blog',
				'impeka_header_nav'  => 'main-menu',
				'impeka_header_second_nav'  => 'second-menu',
				'impeka_responsive_nav'  => 'responsive-menu',
				'featured_plugins' => array( 'gutenberg' ),
				'required_plugins' => array( 'impeka-extension', 'impeka-gutenberg-extension' ),
			),
			array(
				'id'   => 'impeka-musician',
				'name' => esc_html__( 'Impeka Musician', 'impeka-importer' ),
				'dir'  => 'impeka-musician',
				'preview'  => 'impeka/impeka-musician',
				'description'  => esc_html__( 'Scrolling one page case study combined with some of the latest web trends. Created with Impeka Enhanced Page Builder.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'js_composer' ),
				'required_plugins' => array( 'impeka-extension', 'js_composer', 'impeka-wpb-extension' ),
			),
			array(
				'id'   => 'impeka-freelancer',
				'name' => esc_html__( 'Impeka Freelancer', 'impeka-importer' ),
				'dir'  => 'impeka-freelancer',
				'preview'  => 'impeka/impeka-freelancer',
				'description'  => esc_html__( 'Are you a freelancer and want to show off your work professionally? Get inspired with Impeka Freelancer.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'elementor' ),
				'required_plugins' => array( 'impeka-extension', 'elementor', 'impeka-elementor-extension' ),
			),
			array(
				'id'   => 'impeka-minimal-portfolio',
				'name' => esc_html__( 'Impeka Minimal Portfolio', 'impeka-importer' ),
				'dir'  => 'impeka-minimal-portfolio',
				'preview'  => 'impeka/impeka-minimal-portfolio',
				'description'  => esc_html__( 'Simple & clean case study with custom off-canvas menu, impressive animations and page transitions.', 'impeka-importer' ),
				'homepage' => 'Home',
				'featured_plugins' => array( 'js_composer' ),
				'required_plugins' => array( 'impeka-extension', 'js_composer', 'impeka-wpb-extension' ),
				'single_selector' => true,
			),
			array(
				'id'   => 'impeka-creative-portfolio',
				'name' => esc_html__( 'Impeka Creative Portfolio', 'impeka-importer' ),
				'dir'  => 'impeka-creative-portfolio',
				'preview'  => 'impeka/impeka-creative-portfolio',
				'description'  => esc_html__( 'Modern & Creative portfolio case study to showcase your work and get noticed.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'gutenberg' ),
				'required_plugins' => array( 'impeka-extension', 'impeka-gutenberg-extension' ),
			),
			array(
				'id'   => 'impeka-yoga-fitness',
				'name' => esc_html__( 'Impeka Yoga Fitness', 'impeka-importer' ),
				'dir'  => 'impeka-yoga-fitness',
				'preview'  => 'impeka/impeka-yoga-fitness',
				'description'  => esc_html__( 'Its minimal design makes it suitable for personal trainers, fitness studios and many more.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'elementor' ),
				'required_plugins' => array( 'impeka-extension', 'elementor', 'impeka-elementor-extension' ),
			),
			array(
				'id'   => 'impeka-group-1',
				'name' => esc_html__( 'Impeka Modern Corporate', 'impeka-importer' ),
				'dir'  => 'impeka-group-1',
				'preview'  => 'impeka/impeka-group-i',
				'description'  => esc_html__( 'Another detailed case study created with the Elementor page builder. A modern corporate case study, suitable for any industry.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'impeka_header_second_nav'  => 'second-menu',
				'featured_plugins' => array( 'elementor' ),
				'required_plugins' => array( 'impeka-extension', 'elementor', 'impeka-elementor-extension' ),
			),
			array(
				'id'   => 'impeka-group-2',
				'name' => esc_html__( 'Impeka Classic Corporate', 'impeka-importer' ),
				'dir'  => 'impeka-group-2',
				'preview'  => 'impeka/impeka-group-ii',
				'description'  => esc_html__( 'A detailed corporate case study, created with the Elementor page builder. Present your services and engage more visitors.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'elementor' ),
				'required_plugins' => array( 'impeka-extension', 'elementor', 'impeka-elementor-extension' ),
				'widget_selector' => true,
			),
			array(
				'id'   => 'impeka-group-3',
				'name' => esc_html__( 'Impeka Business', 'impeka-importer' ),
				'dir'  => 'impeka-group-3',
				'preview'  => 'impeka/impeka-group-iii',
				'description'  => esc_html__( 'A business demo created with the Elementor page builder. The vast array of theme options makes this perfect for websites of all kinds.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'elementor' ),
				'required_plugins' => array( 'impeka-extension', 'elementor', 'impeka-elementor-extension' ),
			),
			array(
				'id'   => 'impeka-group-4',
				'name' => esc_html__( 'Impeka Creative Corporate', 'impeka-importer' ),
				'dir'  => 'impeka-group-4',
				'preview'  => 'impeka/impeka-group-iv',
				'description'  => esc_html__( 'Another Impeka & Elementor based demo. It\'s bold, comes with minimal design and page transitions. ', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'elementor' ),
				'required_plugins' => array( 'impeka-extension', 'elementor', 'impeka-elementor-extension' ),
			),
			array(
				'id'   => 'impeka-group-6',
				'name' => esc_html__( 'Impeka Agency Corporate', 'impeka-importer' ),
				'dir'  => 'impeka-group-6',
				'preview'  => 'impeka/impeka-group-vi',
				'description'  => esc_html__( 'Another corporate case study created with the Elementor page builder. Most suitable for agencies of any kind.', 'impeka-importer' ),
				'homepage' => 'Home',
				'postspage' => 'Blog',
				'impeka_header_nav'  => 'main-menu',
				'impeka_footer_nav'  => 'footer-menu',
				'featured_plugins' => array( 'elementor' ),
				'required_plugins' => array( 'impeka-extension', 'elementor', 'impeka-elementor-extension' ),
			),
			array(
				'id'   => 'impeka-lawyer',
				'name' => esc_html__( 'Impeka Lawyer', 'impeka-importer' ),
				'dir'  => 'impeka-lawyer',
				'preview'  => 'impeka/impeka-lawyer',
				'description'  => esc_html__( 'A detailed lawyer case study, created with the Elementor page builder. Present your lawyer services and your firm to engage more visitors.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'elementor' ),
				'required_plugins' => array( 'impeka-extension', 'elementor', 'impeka-elementor-extension' ),
			),
			array(
				'id'   => 'impeka-dental-clinic',
				'name' => esc_html__( 'Impeka Dental Clinic', 'impeka-importer' ),
				'dir'  => 'impeka-dental-clinic',
				'preview'  => 'impeka/impeka-dental-clinic',
				'description'  => esc_html__( 'A detailed dental clinic case study, created with the Elementor page builder. Present your services, use forms for online appointments with your patients.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'elementor' ),
				'required_plugins' => array( 'impeka-extension', 'elementor', 'impeka-elementor-extension' ),
				'widget_selector' => true,
			),
			array(
				'id'   => 'impeka-barber',
				'name' => esc_html__( 'Impeka Barber Shop', 'impeka-importer' ),
				'dir'  => 'impeka-barber',
				'preview'  => 'impeka/impeka-barber',
				'description'  => esc_html__( 'Handcrafted for Barbershops, hairstylists and beard barbers. One page case study, created with Impeka Enhanced Page Builder.', 'impeka-importer' ),
				'homepage' => 'Homepage',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'js_composer' ),
				'required_plugins' => array( 'impeka-extension', 'js_composer', 'impeka-wpb-extension' ),
			),
			array(
				'id'   => 'impeka-blog-2',
				'name' => esc_html__( 'Impeka Blog II', 'impeka-importer' ),
				'dir'  => 'impeka-blog-2',
				'preview'  => 'impeka/impeka-blog-ii',
				'description'  => esc_html__( 'Handcrafted for Bloggers and Writers. With a side header layout, a demo was developed with Impeka Enhanced Page Builder.', 'impeka-importer' ),
				'homepage' => 'Home',
				'postspage' => 'Blog',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'js_composer' ),
				'required_plugins' => array( 'impeka-extension', 'js_composer', 'impeka-wpb-extension' ),
			),
			array(
				'id'   => 'impeka-fitness',
				'name' => esc_html__( 'Impeka Fitness', 'impeka-importer' ),
				'dir'  => 'impeka-fitness',
				'preview'  => 'impeka/impeka-fitness',
				'description'  => esc_html__( 'Handcrafted for Personal trainers and Fitness studios. With intro loader, page transitions and impressive layouts created with Impeka Enhanced Page Builder.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'js_composer' ),
				'required_plugins' => array( 'impeka-extension', 'js_composer', 'impeka-wpb-extension' ),
			),
			array(
				'id'   => 'impeka-wedding',
				'name' => esc_html__( 'Impeka Wedding', 'impeka-importer' ),
				'dir'  => 'impeka-wedding',
				'preview'  => 'impeka/impeka-wedding',
				'description'  => esc_html__( 'Handcrafted for Wedding Websites. Clean and simple, yet impressive layouts to share your love story and your wedding details.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'js_composer' ),
				'required_plugins' => array( 'impeka-extension', 'js_composer', 'impeka-wpb-extension' ),
			),
			array(
				'id'   => 'impeka-design-agency',
				'name' => esc_html__( 'Impeka Design Agency', 'impeka-importer' ),
				'dir'  => 'impeka-design-agency',
				'preview'  => 'impeka/impeka-design-agency',
				'description'  => esc_html__( 'Handcrafted for Design Agencies & Freelancers. Impressive off-canvas menu, unique Safe Button and layouts created with Impeka enhanced Page Builder.', 'impeka-importer' ),
				'homepage' => 'Homepage',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'js_composer' ),
				'required_plugins' => array( 'impeka-extension', 'js_composer', 'impeka-wpb-extension' ),
			),
			array(
				'id'   => 'impeka-bakery',
				'name' => esc_html__( 'Impeka Bakery', 'impeka-importer' ),
				'dir'  => 'impeka-bakery',
				'preview'  => 'impeka/impeka-bakery',
				'description'  => esc_html__( "Clean and simple, your Bakery shop has to feel fresh and unique without copying another bakery. Created with Impeka's enhanced Page Builder.", 'impeka-importer' ),
				'homepage' => 'Homepage',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'js_composer' ),
				'required_plugins' => array( 'impeka-extension', 'js_composer', 'impeka-wpb-extension' ),
			),
			array(
				'id'   => 'impeka-branding-agency',
				'name' => esc_html__( 'Impeka Branding Agency', 'impeka-importer' ),
				'dir'  => 'impeka-branding-agency',
				'preview'  => 'impeka/impeka-branding-agency',
				'description'  => esc_html__( "Clean and simple, your Agency has to feel fresh and unique. Created with Impeka's enhanced Page Builder.", 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'js_composer' ),
				'required_plugins' => array( 'impeka-extension', 'js_composer', 'impeka-wpb-extension' ),
			),
			array(
				'id'   => 'impeka-corporate-creative',
				'name' => esc_html__( 'Impeka Corporate Creative', 'impeka-importer' ),
				'dir'  => 'impeka-corporate-creative',
				'preview'  => 'impeka/impeka-corporate-creative',
				'description'  => esc_html__( "Clean and modern with impressive clipping animations to increase user Interaction on your Website.", 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'js_composer' ),
				'required_plugins' => array( 'impeka-extension', 'js_composer', 'impeka-wpb-extension' ),
			),
			array(
				'id'   => 'impeka-agency',
				'name' => esc_html__( 'Impeka Agency', 'impeka-importer' ),
				'dir'  => 'impeka-agency',
				'preview'  => 'impeka/impeka-agency',
				'description'  => esc_html__( 'Handcrafted for Design Agencies & Freelancers. Impressive off-canvas menu, minimal & clean layouts, created with Elementor Builder.', 'impeka-importer' ),
				'homepage' => 'Home',
				'postspage' => 'Insights',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'elementor' ),
				'required_plugins' => array( 'impeka-extension', 'elementor', 'impeka-elementor-extension' ),
			),
			array(
				'id'   => 'impeka-startup',
				'name' => esc_html__( 'Impeka Startup', 'impeka-importer' ),
				'dir'  => 'impeka-startup',
				'preview'  => 'impeka/impeka-startup',
				'description'  => esc_html__( 'Handcrafted for Startups. Minimal & clean layouts, blog posts page for your insights. Another demo created with Elementor Builder.', 'impeka-importer' ),
				'homepage' => 'Home',
				'postspage' => 'Insights',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'elementor' ),
				'required_plugins' => array( 'impeka-extension', 'elementor', 'impeka-elementor-extension' ),
			),
			array(
				'id'   => 'impeka-spa',
				'name' => esc_html__( 'Impeka Spa', 'impeka-importer' ),
				'dir'  => 'impeka-spa',
				'preview'  => 'impeka/impeka-spa',
				'description'  => esc_html__( 'Demonstrate your luxury spa services or any other relaxing services. Another demo created with Elementor Builder.', 'impeka-importer' ),
				'homepage' => 'Home',
				'impeka_header_nav'  => 'main-menu',
				'featured_plugins' => array( 'elementor' ),
				'required_plugins' => array( 'impeka-extension', 'elementor', 'impeka-elementor-extension' ),
			),
		);

		return $grve_dummy_data_list;
	}

	/**
	 * Imports theme options from file ( Redux Framework )
 	 */
	function grve_import_options( $import_file, $dummy_id = 'impeka'  ) {
		global $impeka_grve_redux_framework;

		$import_array = file_get_contents( $import_file );

		if ( !empty( $import_array ) ) {

			$default_logo_path = get_template_directory_uri() .'/images/logos/';
			$logo_path = get_template_directory_uri() .'/images/logos/' . $dummy_id . '/';

			$imported_options = array();
			$imported_options = json_decode( htmlspecialchars_decode( $import_array ), true );

			$imported_options['logo_dark'] = array( 'url' => $default_logo_path . 'logo-dark.png', 'width' => '198', 'height' => '60', 'id' => "" );
			$imported_options['logo_light'] = array( 'url' => $default_logo_path . 'logo-light.png', 'width' => '198', 'height' => '60', 'id' => "" );
			$imported_options['logo_responsive'] = $imported_options['logo_sticky'] = $imported_options['logo'] = $imported_options['logo_dark'];

			//Reset to default
			$imported_options['logo_height'] = '30';
			$imported_options['logo_small_screens_height'] = '24';
			$imported_options['header_sticky_shrink_logo_height'] = '30';
			$imported_options['responsive_logo_height'] = '24';
			$imported_options['gmap_api_key'] = '';
			$imported_options['print_emoji'] = '1';
			$imported_options['css_generation'] = '0';

			foreach($imported_options as $key => $value) {
				$impeka_grve_redux_framework->ReduxFramework->set($key, $value);
			}

		} else {
			return false;
		}

		return true;
	}
}

//Omit closing PHP tag to avoid accidental whitespace output errors.
