<?php
if ( ! defined ( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

if ( ! class_exists ( 'WC_Product_Gift_Card' ) ) {

	/**
	 *
	 * @class   YITH_YWGC_Gift_Card
	 *
	 * @since   1.0.0
	 * @author  Lorenzo Giuffrida
	 */
	class WC_Product_Gift_Card extends WC_Product {

		const YWGC_AMOUNTS = '_gift_card_amounts';
		const YWGC_OVERRIDE_GLOBAL_SETTINGS = '_ywgc_override_global_settings';
		const YWGC_MANUAL_AMOUNT_MODE = '_ywgc_manual_amount_mode';
		const YWGC_ADD_DISCOUNT_SETTINGS = '_ywgc_add_discount_settings';
		const YWGC_EXPIRATION_SETTINGS = '_ywgc_expiration_settings';
		const YWGC_PRODUCT_IMAGE = '_ywgc_product_image';
		const YWGC_PRODUCT_TEMPLATE_DESIGN = '_ywgc_show_product_template_design';

		public $amounts = null;

		/**
		 * Initialize a gift card product.
		 *
		 * @param mixed $product
		 */
		public function __construct( $product ) {
			parent::__construct( $product );

			$this->downloadable = 'no';
			$this->product_type = YWGC_GIFT_CARD_PRODUCT_TYPE;
		}

		public function get_type() {
			return YWGC_GIFT_CARD_PRODUCT_TYPE;
		}

		/**
		 *
		 * @return bool
		 */
		public function is_downloadable() {
			return false;
		}

		/**
		 * Retrieve the number of current amounts for this product
		 *
		 * @return int
		 * @author Lorenzo Giuffrida
		 * @since  1.0.0
		 */
		public function get_amounts_count() {
			$amounts = $this->get_product_amounts ();

			return count ( $amounts );
		}

		/**
		 * Retrieve the amounts set for the product
		 * @return array
		 */
		public function get_product_amounts() {

            if ( ! is_array ( $this->amounts ) ) {
                if ( $this->id ) {
	                $result = get_post_meta( $this->get_id(), self::YWGC_AMOUNTS, true );
                    $this->amounts = is_array ( $result ) ? $result : array();
                }
            }

			return apply_filters ( 'yith_ywgc_gift_card_amounts', array_values($this->amounts), $this );
		}

		/**
		 * Retrieve the amounts set for the product in the different currencies
		 * @return array
		 */
		public function get_product_amounts_by_currency( $currency_id ) {

			$amounts_by_currency_meta = '_gift_card_amounts_' . $currency_id;

			if ( $this->id ) {
				$result = get_post_meta( $this->get_id(), $amounts_by_currency_meta, true );

				$currency_amounts = is_array ( $result ) ? $result : $this->get_product_amounts_converted_to_currency( $currency_id );

				$currency_amounts = array_values( $currency_amounts );
			}
			else{
				$currency_amounts = array();
			}

			return apply_filters ( 'yith_ywgc_gift_card_amounts_by_currency', array_values($currency_amounts), $this, $currency_id );
		}

		/**
		 * Retrieve the amounts set for the product converted to the different currencies
		 * @return array
		 */
		public function get_product_amounts_converted_to_currency( $currency_id ) {

			$converted_amounts = array();
			$product_amounts = $this->get_product_amounts();

			foreach ( $product_amounts as $amount ){

				$converted_amounts[] = apply_filters( 'yith_wcmcs_convert_price', $amount, array(
					'from' => yith_wcmcs_get_current_currency_id(),
					'to'   => $currency_id,
				) );

			}

			return $converted_amounts;
		}

		/**
		 * Returns false if the product cannot be bought.
		 *
		 * @return bool
		 */
		public function is_purchasable() {

			$purchasable = $this->get_amounts_count() > 0 || $this->is_manual_amount_enabled () ? 1 : 0;

			return apply_filters ( 'woocommerce_is_purchasable', $purchasable, $this );
		}

		/**
		 * Save current gift card amounts
		 *
		 * @param array $amounts
		 */
		public function set_amounts( $amounts = array() ) {
			$this->amounts = $amounts;
		}

		/**
		 * Save current gift card amounts
		 *
		 * @param array $amounts
		 */
		public function save_amounts( $amounts = array(), $currency_id = '' ) {

			if ( $currency_id != '' ){
				$amounts_by_currency_meta = '_gift_card_amounts_' . $currency_id;

				update_post_meta( $this->get_id(), $amounts_by_currency_meta, $amounts );

			}
			else{

				update_post_meta( $this->get_id(), self::YWGC_AMOUNTS, $amounts );
			}
		}

		/**
		 * Update the design status for the gift card
		 *
		 * @param $status
		 */
		public function set_design_status( $status ) {
			yit_save_prop ( $this, self::YWGC_PRODUCT_TEMPLATE_DESIGN, $status );
		}

		/**
		 * Retrieve the design status
		 *
		 * @return mixed
		 */
		public function get_design_status() {

			return yit_get_prop ( $this->get_product_instance (), self::YWGC_PRODUCT_TEMPLATE_DESIGN );
		}

		public function update_override_global_settings_status( $status ) {
			yit_save_prop ( $this, self::YWGC_OVERRIDE_GLOBAL_SETTINGS, $status );
		}

		/**
		 * Update the manual amount status.
		 * Available values are "global", "accept" and "reject"
		 *
		 * @param string $status
		 */
		public function update_manual_amount_status( $status ) {
			yit_save_prop ( $this, self::YWGC_MANUAL_AMOUNT_MODE, $status );
		}


		public function update_add_discount_settings_status( $status ) {
			yit_save_prop ( $this, self::YWGC_ADD_DISCOUNT_SETTINGS, $status );
		}

		public function update_expiration_settings_status( $status ) {
			yit_save_prop ( $this, self::YWGC_EXPIRATION_SETTINGS, $status );
		}

		/**
		 * Process the current product instance in order to let third party plugin
		 * change the reference(Useful for WPML and similar plugins)
		 *
		 * @return WC_Product
		 */
		protected function get_product_instance() {

			return apply_filters ( 'yith_ywgc_get_product_instance', $this );
		}

		public function get_override_global_settings_status() {

			return yit_get_prop ( $this->get_product_instance (), self::YWGC_OVERRIDE_GLOBAL_SETTINGS );
		}

		public function get_add_discount_settings_status() {

			return yit_get_prop ( $this->get_product_instance (), self::YWGC_ADD_DISCOUNT_SETTINGS );
		}

		public function get_expiration_settings_status() {

			return yit_get_prop ( $this->get_product_instance (), self::YWGC_EXPIRATION_SETTINGS );
		}

		/**
		 * Retrieve the manual amount status for this product.
		 *
		 * Available values are "global", "accept" and "reject"
		 * @return mixed
		 */
		public function get_manual_amount_status() {

			return yit_get_prop ( $this->get_product_instance (), self::YWGC_MANUAL_AMOUNT_MODE );
		}

		/**
		 * Retrieve if manual amount is enabled for this gift card
		 */
		public function is_manual_amount_enabled() {

			$override_globals = $this->get_override_global_settings_status();
			$status = $this->get_manual_amount_status ();

			if ( $override_globals == false ){
				$status = get_option ( 'ywgc_permit_free_amount', 'no' );
			}

			if ( $status == 'yes' || $status == 1 ){
				$status =  'yes';
			}

			return apply_filters ( 'yith_gift_cards_is_manual_amount_enabled', 'yes' == $status, $status, $this );
		}

		/**
		 * Returns the price in html format
		 *
		 * @access public
		 *
		 * @param string $price (default: '')
		 *
		 * @return string
		 */
		public function get_price_html( $price = '' ) {
			$amounts = $this->get_amounts_to_be_shown ();

			// No price for current gift card
			if ( ! count ( $amounts ) ) {
				$price = apply_filters ( 'yith_woocommerce_gift_cards_empty_price_html', '', $this );
			} else {
				ksort ( $amounts, SORT_NUMERIC );

				$min_price = current ( $amounts );
				$min_price = wc_price( $min_price['price'] );
				$max_price = end ( $amounts );
				$max_price = wc_price( $max_price['price'] );

				$price = apply_filters( 'yith_woocommerce_gift_cards_amount_range', $min_price !== $max_price ? wc_format_price_range( $min_price, $max_price ) : $min_price, $this, $min_price, $max_price );
			}

			return apply_filters ( 'woocommerce_get_price_html', $price, $this );
		}

		/**
		 * Retrieve an array of gift cards amounts with the corrected value to be shown(inclusive or not inclusive taxes)
		 *
		 * @return array
		 * @author Lorenzo Giuffrida
		 * @since  1.0.0
		 */
		public function get_amounts_to_be_shown() {

			$customer_currency_id = class_exists( 'YITH_WCMCS_Currency_Handler') ? YITH_WCMCS_Currency_Handler::get_instance()->get_client_currency_id() : get_woocommerce_currency();

			$default_currency_id = function_exists( 'yith_wcmcs_get_wc_currency_options') ? yith_wcmcs_get_wc_currency_options( 'currency' ) : get_woocommerce_currency();

			$amounts_to_show  = array();

			$tax_display_mode = get_option ( 'woocommerce_tax_display_shop' );

			$custom_prices_enabled = 'yes' === get_post_meta( $this->id, '_ywgc_wcmcs_custom_prices', true );

			if ( $customer_currency_id != $default_currency_id && $custom_prices_enabled ){
				$display_amounts = $this->get_product_amounts_by_currency( $customer_currency_id );
				$product_amounts = $this->get_product_amounts_by_currency( $customer_currency_id );
			}
			else{
				$display_amounts = $this->get_product_amounts();
				$product_amounts = $this->get_product_amounts();
			}

			$index = 0;
			foreach ( $display_amounts as $amount ) {

				$amount = wc_format_decimal( $amount );

				if ( 'incl' === $tax_display_mode ) {
					$price = yit_get_price_including_tax ( $this, 1, $amount );
				} else {
					$price = yit_get_price_excluding_tax ( $this, 1, $amount );
				}

				$original_amount = $product_amounts[ $index ];

				$negative        = $price < 0;
				$price_format    = get_woocommerce_price_format ();
				$formatted_price = ( $negative ? '-' : '' ) . sprintf ( $price_format, get_woocommerce_currency_symbol (), $price );

				$amounts_to_show[ $original_amount ] = array(
				    'amount'    => $amount,
					'price'     => $price,
					'wc-price'  => $formatted_price,
					'title'     => wc_price ( $price )
				);
				$index ++;
			}

			return apply_filters ( 'yith_ywgc_gift_cards_amounts', $amounts_to_show, $this->id );
		}

		/**
		 * Get the add to cart button text
		 *
		 * @return string
		 */
		public function add_to_cart_text() {
            $text = $this->is_purchasable () && $this->is_in_stock () ? apply_filters( 'yith_ywgc_select_amount_text' , esc_html__( 'Select amount', 'yith-woocommerce-gift-cards' ) ) : esc_html__( 'Read more', 'yith-woocommerce-gift-cards' );

			return apply_filters ( 'woocommerce_product_add_to_cart_text', $text, $this );
		}

		/**
		 * Add a new amount to the gift cards
		 *
		 * @param float $amount
		 * @param string $currency_id
		 *
		 * @return bool
		 * @author Lorenzo Giuffrida
		 * @since  1.0.0
		 */
		public function add_amount( $amount, $currency_id = '' ) {

			if ( $currency_id != ''){
				$product_amounts = $this->get_product_amounts_by_currency( $currency_id );
			}
			else{
				$product_amounts = $this->get_product_amounts ();
			}

			if ( ! in_array ( $amount, $product_amounts ) ) {

				$product_amounts[] = $amount;
				sort ( $product_amounts, SORT_NUMERIC );
				$this->save_amounts ( $product_amounts, $currency_id );

				return true;
			}
			else{
				sort ( $product_amounts, SORT_NUMERIC );
				$this->save_amounts ( $product_amounts, $currency_id );

				return true;
			}

			return false;
		}

		/**
		 * Remove an amount from the amounts list
		 *
		 * @param float $amount
		 *
		 * @return bool
		 * @author Lorenzo Giuffrida
		 * @since  1.0.0
		 */
		public function remove_amount( $amount, $index, $currency_id = '' ) {

			if ( $currency_id != ''){
				$product_amounts = $this->get_product_amounts_by_currency( $currency_id );
			}
			else{
				$product_amounts = $this->get_product_amounts ();
			}

			if ( array_key_exists( $index, $product_amounts) ) {

				unset( $product_amounts[ $index ] );

				$this->save_amounts ( $product_amounts, $currency_id );

				return true;
			}

			return false;
		}

		/**
		 * Update an amount from the amounts list
		 *
		 * @param float $amount
		 *
		 * @return bool
		 * @author Lorenzo Giuffrida
		 * @since  1.0.0
		 */
		public function update_amount( $amount, $prev_amount, $currency_id ) {

			if ( $currency_id != ''){
				$product_amounts = $this->get_product_amounts_by_currency( $currency_id );
			}
			else{
				$product_amounts = $this->get_product_amounts ();
			}

			if ( ! in_array ( $amount, $product_amounts ) ) {

				$key = array_search ( $prev_amount, $product_amounts );
				$product_amounts[$key] = $amount;
				sort ( $product_amounts, SORT_NUMERIC );
				//$this->save_amounts ( $product_amounts, $currency_id );

				return true;
			}
			else{
				sort ( $product_amounts, SORT_NUMERIC );
				//$this->save_amounts ( $product_amounts, $currency_id );

				return true;
			}

			return false;

		}




		/**
		 * Retrieve the custom image set from the edit product page for a specific gift card product
		 *
		 * @param string $size
		 * @param string $return Choose whether to return url or id (url|id)
		 *
		 * @return mixed
		 */
		public function get_manual_header_image( $size = 'full', $return = 'url' ) {
            global $post;
            $image_url = '';

			$product = wc_get_product($post);

			if ( !is_object($product))
			    return;

			if ( $product  ) {
			    $image_id = yit_get_prop ( $product, self::YWGC_PRODUCT_IMAGE );
            }
            else{
                $image_id = '';
            }

            $image_id = ( isset( $image_id ) && $image_id ) ? $image_id : yit_get_prop ( $this->get_product_instance(), self::YWGC_PRODUCT_IMAGE );


			$image_id = ( $image_id != '' ) ? $image_id : get_post_thumbnail_id( $post->ID );

			if ( $return == 'id' ) {
				return $image_id;
			}

			if ( $image_id ) {
				$image     = wp_get_attachment_image_src ( $image_id, $size );
				$image_url = $image[0];
			}

			return $image_url;
		}

		/**
		 * Set the header image for a gift card product
		 *
		 * @param int $attachment_id
		 */
		public function set_header_image( $attachment_id ) {

			yit_save_prop ( $this, self::YWGC_PRODUCT_IMAGE, $attachment_id );
		}

		/**
		 * Unset the header image for a gift card product
		 *
		 */
		public function unset_header_image() {

			yit_delete_prop ( $this, self::YWGC_PRODUCT_IMAGE );
		}
	}
}
